////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////
package com.crankuptheamps.client;

import java.io.IOException;
import java.util.Arrays;

import com.crankuptheamps.client.BlockPublishStore.ByteSequence;

/**
 * ArrayStoreBuffer is a simple wrapper for a java array that maintains position.
 * Very similar to the MemoryStoreBuffer, but instead of using a ByteBuffer, 
 * it just uses an array directly.  Resize may be a tad more expensive, but
 * steady-state latencies are significantly lower.
 *
 */
public class ArrayStoreBuffer implements BlockPublishStore.Buffer
{
    volatile byte[] _buf = null;
    volatile int _pos = 0;
    PublishStoreResizeHandler _resizeHandler = null;
    Store _store = null;
    
    public byte[] getBuffer() { return _buf; }
    
    public long getSize() throws IOException
    {
        return _buf==null?0:_buf.length;
    }
    
    public void setSize(long newSize) throws IOException
    {
        if (_resizeHandler != null && !_resizeHandler.invoke(_store, newSize))
            return;
        byte[] oldBuf = _buf;
        if(oldBuf == null || newSize > oldBuf.length)
        {
            byte[] newBuf = new byte[(int)newSize];

            if(oldBuf != null && oldBuf.length > 0)
            {
                System.arraycopy(oldBuf, 0, newBuf, 0, oldBuf.length);
            }
            _buf = newBuf;
        }
    }

    public long getPosition() throws IOException
    {
        return _pos;
    }

    public void setPosition(long position) throws IOException
    {
        _pos = (int)position;
        
    }

    public void putByte(byte b) throws IOException
    {
        _buf[_pos++] = b;
        
    }

    public byte getByte() throws IOException
    {
        return _buf[_pos++];
    }

    public void putInt(int i) throws IOException
    {
        _buf[_pos++] = (byte)((i & 0xFF000000) >> 24);
        _buf[_pos++] = (byte)((i & 0xFF0000) >> 16);
        _buf[_pos++] = (byte)((i & 0xFF00) >> 8);
        _buf[_pos++] = (byte)(i & 0xFF);
        
        
    }

    public int getInt() throws IOException
    {
        int result = _buf[_pos+3] & 0xff |
                ((_buf[_pos+2]&0xff) << 8) |
                ((_buf[_pos+1]&0xff) << 16) |
                ((_buf[_pos+0]&0xff) << 24);
        _pos += 4;
        return result;
    }

    public void putLong(long l) throws IOException
    {
        _buf[_pos++] = (byte)((l & 0xFF00000000000000L) >> 56);
        _buf[_pos++] = (byte)((l & 0xFF000000000000L) >> 48);
        _buf[_pos++] = (byte)((l & 0xFF0000000000L) >> 40);
        _buf[_pos++] = (byte)((l & 0xFF00000000L) >> 32);
        _buf[_pos++] = (byte)((l & 0xFF000000L) >> 24);
        _buf[_pos++] = (byte)((l & 0xFF0000L) >> 16);
        _buf[_pos++] = (byte)((l & 0xFF00L) >> 8);
        _buf[_pos++] = (byte)(l & 0xFFL);
        
    }

    public long getLong() throws IOException
    {
        long result = _buf[_pos+7] & 0xff |
                ((_buf[_pos+6]&0xff) << 8) |
                ((_buf[_pos+5]&0xff) << 16) |
                ((_buf[_pos+4]&0xff) << 24) |
                ((_buf[_pos+3]&0xff) << 32) |
                ((_buf[_pos+2]&0xff) << 40) |
                ((_buf[_pos+1]&0xff) << 48) |
                ((_buf[_pos+0]&0xff) << 56);
        _pos += 8;
        return result; 
    }

    public void putBytes(ByteSequence bytes) throws IOException
    {
        System.arraycopy(bytes.array, (int)bytes.offset, _buf, _pos, (int)bytes.len);
        _pos += bytes.len;
    }

    public void getBytes(ByteSequence outBytes) throws IOException
    {
        outBytes.array = _buf;
        outBytes.offset = _pos;
        _pos += outBytes.len;
        
    }

    public void zero(long offset, int length) throws IOException
    {
        Arrays.fill(_buf, (int)offset, (int)offset+length, (byte)0);
    }

    public void setResizeHandler(PublishStoreResizeHandler handler, Store store)
    {
        _resizeHandler = handler;
        _store = store;
    }

}
