////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;

public interface BookmarkStore
{
    /**
      * Log a bookmark to the persistent log.
      * @return the corresponding bookmark sequence number
      *         for this bookmark
      */
    long log(Message message) throws AMPSException;

    /**
      * Log a discard-bookmark entry to the persistent log
      * based on a bookmark sequence number.
      * Recommended to use discard(Message) instead for performance.
      *
      */
    void discard(Field subId, long bookmarkSeqNo) throws AMPSException;

    /**
     * Log a discard-bookmark entry to the persistent log
     * based on a Message.
     */
    void discard(Message message) throws AMPSException;
    
    /**
      * Returns the most recent bookmark from the log that
      * ought to be used for (re-)subscriptions.
      */
    Field getMostRecent(Field subID) throws AMPSException;

    /**
      * Called for each arriving message to determine if
      * the application has already seen this bookmark and should
      * not be reprocessed.  Returns 'true' if the bookmark is
      * in the log and should not be re-processed, false otherwise.
      */
    boolean isDiscarded(Message message) throws AMPSException;
    
    /**
     * Called to purge the contents of this store.
     * Removes any tracking history associated with publishers and received
     * messages, and may delete or truncate on-disk representations as well.
     */
    void purge() throws AMPSException;

    /**
     * Called to set a resize handler that is invoked when the store needs to resize.
     */
    public void setResizeHandler(BookmarkStoreResizeHandler handler);

    /**
     * Called to retrieve the sequence number of the oldest bookmark in the store.
     */
    long getOldestBookmarkSeq(Field subId) throws AMPSException;

    /**
     * Called to mark a message as safely persisted by AMPS to all of its sync
     * replication destination.
     */
    void persisted(Field subId, BookmarkField bookmark) throws AMPSException;

    /**
     * Called to mark a message as safely persisted by AMPS to all of its sync
     * replication destination.
     */
    void persisted(Field subId, long bookmarkSeqNo) throws AMPSException;

    /**
     * Called internally to let the bookmark store know that the AMPS server
     * version doesn't support persisted acks for bookmarks.
     */
    void noPersistedAcks(Field subId) throws AMPSException;

    /**
     * Called by the Client when connected to an AMPS server to indicate
     * what version the server is.
     */
    void setServerVersion(int version_);

}
