////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.beans.ExceptionListener;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.lang.ClassLoader;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Properties;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.jar.Attributes;
import java.util.jar.Manifest;

import com.crankuptheamps.client.Store.StoreReplayer;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.exception.AlreadyConnectedException;
import com.crankuptheamps.client.exception.AuthenticationException;
import com.crankuptheamps.client.exception.BadFilterException;
import com.crankuptheamps.client.exception.BadRegexTopicException;
import com.crankuptheamps.client.exception.CommandException;
import com.crankuptheamps.client.exception.ConnectionException;
import com.crankuptheamps.client.exception.ConnectionRefusedException;
import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.InvalidTopicException;
import com.crankuptheamps.client.exception.InvalidURIException;
import com.crankuptheamps.client.exception.ProtocolException;
import com.crankuptheamps.client.exception.NotEntitledException;
import com.crankuptheamps.client.exception.NameInUseException;
import com.crankuptheamps.client.exception.RetryOperationException;
import com.crankuptheamps.client.exception.SubidInUseException;
import com.crankuptheamps.client.exception.SubscriptionAlreadyExistsException;
import com.crankuptheamps.client.exception.TimedOutException;
import com.crankuptheamps.client.exception.TransportTypeException;
import com.crankuptheamps.client.exception.StoreException;
import com.crankuptheamps.client.exception.UnknownException;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;
import com.crankuptheamps.client.fields.LongField;
import com.crankuptheamps.client.fields.StringField;

/**
 *   The base AMPS client object used in AMPS applications. Each client object
 *   manages a single connection to AMPS. Each AMPS connection has a name,
 *   a specific transport (such as tcp), a protocol (used for framing messages
 *   to AMPS), and a message type (such as FIX or JSON).
 *   <p>
 *   The client object creates and manages background threads. The object
 *   provides both a synchronous interface for processing messages on the
 *   calling thread and an asynchronous interface suitable for populating
 *   a queue to be processed by worker threads.
 *   <p>
 *   The Client provides named convenience methods for many AMPS commands.
 *   These methods provide control over commonly used parameters and support
 *   most programs. For full control over AMPS, you build a
 *   {@link com.crankuptheamps.client.Command} object and use
 *   {@link Client#execute } or {@link Client#executeAsync} to run the command.
 *   <p>
 *   AMPS uses the client name to detect duplicate messages, so the name
 *   of each instance of the client application should be unique.
 *   <p>
 *   An example of a simple Java program publishing the JSON message
 *   <code>{ "message" : "Hello, World!"}</code> is listed below.
 *   <pre><code>
 *    public static void main(String[] args) {
 *    Client client = new Client("ConsolePublisher");
 *
 *   try
 *   {
 *     client.connect("tcp://127.0.0.1:9007/amps");
 *     System.out.println("connected..");
 *     client.logon();
 *    
 *     client.publish("messages", "{\"message\" : \"Hello, World!\"}");
 *
 *     System.exit(0);
 *    }
 *   catch (AMPSException e)
 *   {
 *     System.err.println(e.getLocalizedMessage());
 *     e.printStackTrace(System.err);
 *   }
 *
 *  }
 * </code></pre>
 *
 */
public class Client implements java.io.Closeable
{
    private volatile String                 name              = null;
    private Transport                       transport         = null;

    // Please do not change this name: there are tests that use reflection and depend on it.
    private long                            haSequenceNumber  = 1;

    private Store                           publishStore      = null;
    private volatile URI                             uri               = null;
    private volatile ClientDisconnectHandler         disconnectHandler = new DefaultDisconnectHandler();
    protected ExceptionListener             exceptionListener = null;
    private Message                         message           = null;
    private Command                         command = new Command();
    private Message                         beatMessage       = null;
    private final Lock                      lock              = new ReentrantLock();
    //private final Lock                      lock              = new DebugLock("HAClient");

    // Asynchronous->Synchronous ack handling
    private final Condition                 ackReceived  = lock.newCondition();
    private final HashSet<ConnectionStateListener> _connectionStateListeners = new HashSet<ConnectionStateListener>();
    
    static class AckResponse
    {
        public volatile int connectionVersion;
        public volatile int serverVersion;
        public volatile boolean responded;
        public volatile boolean abandoned;
        public volatile int state;
        public volatile int reason;
        public volatile String reasonText;
        public volatile String username;
        public volatile String password;
        public volatile long sequence;
    }

    private Map<CommandId, AckResponse> _acks = new ConcurrentHashMap<CommandId, AckResponse>();
    private MessageRouter _routes = new MessageRouter();
    private ClientHandler                   _handler           = new ClientHandler(this);
    private MessageHandler                  lastChanceMessageHandler = new DefaultMessageHandler();
    private MessageHandler                  duplicateMessageHandler = new DefaultMessageHandler();
    private int                             version           = Version.AMPS_3;
    private String _version = null;
    private volatile BookmarkStore bookmarkStore = new DefaultBookmarkStore();
    private volatile String username;
    private volatile SubscriptionManager subscriptionManager = new DefaultSubscriptionManager();
    private volatile boolean _badTimeToHAPublish = false;
    private volatile FailedWriteHandler _failedWriteHandler = null;
    private int                             serverVersion     = 0;
    private StopWatch heartbeatTimer = new StopWatch();
    public static final int MIN_PERSISTED_BOOKMARK_VERSION = 3080000;
    public static final int MIN_MULTI_BOOKMARK_VERSION = 4000000;
    private static final int MIN_FLUSH_VERSION = 4000000;

    public static final class Version
    {
        public static final int AMPS_1 = 0x00010000;
        public static final int AMPS_2 = 0x00020000;
        public static final int AMPS_3 = 0x00030000;
    }


    /**
     * Functions that take a bookmark (String) such as
     * {@link Client#bookmarkSubscribe} can be passed a literal bookmark ID,
     * or one of these special values.
     *
     * For example, to begin a bookmark subscription at the beginning of the
     * journal, provide {@code Bookmarks.EPOCH} as the bookmark in the 
     * {@link Command} object or the call to
     * bookmarkSubscribe.
     */
    public static final class Bookmarks
    {
        /**
	 * Start the subscription at the first undiscarded message in the
	 * bookmark store, or at the end of the bookmark store if all
	 * messages have been discarded.
         */
        public static final String MOST_RECENT = "recent";

        /**
         * Start the subscription at the beginning of the journal.
         */
        public static final String EPOCH = "0";

        /**
         * Start the subscription at the point in time when AMPS processes
         * the subscription.
         */
        public static final String NOW = "0|1|";
    }

    /**
     * Creates a client.
     *
     * @param name Name for the client. This name is used for duplicate
     * message detection and should be unique.
     */
    public Client(String name)
    {
        this(name, null);
    }

    /**
     * Creates a client.
     *
     * @param name Name for the client. This name is used for duplicate
     * message detection and should be unique.
     * @param version Server version connecting to
     */
    public Client(String name, int version)
    {
        this(name, null, version);
    }

    /**
     * Creates a client with a transport.
     *
     * @param name Name for the client. This name is used for duplicate
     * message detection and should be unique.
     * @param transport Transport to use for this client
     */
    public Client(String name, Transport transport)
    {
        this(name, transport, Version.AMPS_3);
    }

    /**
     * Creates a client with a transport
     *
     * @param name Name for the client. This name is used for duplicate
     * message detection and should be unique.
     * @param transport Transport to use with the client
     * @param version Server version connecting to
     */
    public Client(String name, Transport transport, int version)
    {
        this.name = name;
        if (transport != null)
        {
            this.transport = transport;
            this.transport.setMessageHandler(this._handler);
            this.transport.setDisconnectHandler(this._handler);
            this.message = this.transport.allocateMessage();
            this.beatMessage = this.transport.allocateMessage();
            this.beatMessage.setCommand(Message.Command.Heartbeat);
            this.beatMessage.setOptions("beat");
        }
        this.version = version;
    }
    /**
     * Return the name of the Client.
     *
     * @return String Client name
     */
    public String getName()
    {
        return this.name;
    }

    /**
     * Return the URI the Client is connected to.
     *
     * @return URI
     */
    public URI getURI()
    {
        return this.uri;
    }

    /**
     * Return the server version retrieved during logon. If the client has
     * not logged on or is connected to a server whose version is less than
     * 3.8.0.0 this function will return 0.
     * The version uses 2 digits each for major minor maintenance and hotfix
     * i.e., 3.8.1.5 will return 3080105
     * Versions of AMPS prior to 3.8.0.0 did not return the server version
     * to the client in the logon ack.
     * @return The version of the server as a long.
     */
    public int getServerVersion()
    {
        return this.serverVersion;
    }

    /**
     * Return the numeric value for the given version string with the pattern:
     * Major.Minor.Maintenance.Hotfix
     * The version uses 2 digits each for major minor maintenance and hotfix
     * i.e., 3.8.1.5 will return 3080105
     * Version strings passed in can be shortened to not include all levels
     * so 3.8 will return 3080000.
     * @param version The version string to convert.
     * @return The version as a int.
     * @throws CommandException The string doesn't represent a valid version.
     */
    static public int getVersionAsInt(String version) throws CommandException
    {
        if (version == null || version.length() == 0)
            return 0;

        int retVersion = 0;
        int c = 0;

        int dots = 0;
        int lastDot = -1;
        byte[] bytes = version.getBytes();
        for (int i=0; i<version.length() && dots<4; ++i)
        {
            c = bytes[i];
            if (c == '.')
            {
                ++dots;
                retVersion *= 10;
                if (i-lastDot > 3)
                {
                    throw new CommandException("Too many digits between dots found translating version string.");
                }
                else if (i-lastDot == 3)
                {
                    retVersion += bytes[i-2] - '0';
                }
                retVersion *= 10;
                retVersion += bytes[i-1] - '0';
                lastDot = i;
            }
            else if (c < '0' || c > '9')
            {
                throw new CommandException("Invalid character found in version string.");
            }
            else if (i == version.length()-1)
            {
                ++dots;
                retVersion *= 10;
                if (i-lastDot > 2)
                {
                    throw new CommandException("Too many digits between dots found translating version string.");
                }
                else if (i-lastDot == 2)
                {
                    retVersion += bytes[i-1] - '0';
                }
                retVersion *= 10;
                retVersion += bytes[i] - '0';
            }
        }
        for (; dots<4; ++dots) retVersion *= 100;
        return retVersion;
    }

    /**
     * Return the underlying transport.
     * For advanced users, this method provides direct access to the transport.
     * Care needs to be taken to not modify the underlying transport in ways that
     * are incompatible with the Client.
     *
     * @return Transport Underlying transport instance
     */
    public Transport getTransport()
    {
        try
        {
            lock.lock();
            return this.transport;
        }
        finally
        {
            lock.unlock();
        }
    }

   /**
    * Sets the underlying bookmark store, which is used to track
    * which messages the client has received and which messages have
    * been processed by the program. This method replaces any existing
    * bookmark store for the client, without transferring the state of
    * that store.
    *
    * @param val The new bookmark store.
    */

    public void setBookmarkStore(BookmarkStore val)
    {
        this.bookmarkStore = val;
        if (bookmarkStore != null)
        {
            bookmarkStore.setServerVersion(this.serverVersion);
        }
    }

    /**
     * Returns the underlying bookmark store for this client. The bookmark store
     * is used to track which messages the client has received and which
     * messages have been processed by the program.
     *
     * @return BookmarkStore The bookmark store for this client.
     */
    public BookmarkStore getBookmarkStore()
    {
        return this.bookmarkStore;
    }

    /**
     * Sets the underlying publish store, which is used to store
     * published messages until the AMPS instance has acknowledged
     * those messages. This method replaces any existing
     * publish store for the client, without transferring the state
     * of that store.
     *
     * @param store The new publish store.
     */
    public void setPublishStore(Store store)
    {
        this.publishStore = store;
    }

    /**
     * Returns the underlying publish store for this client. The publish store
     * is used to store published messages until the AMPS instance has
     * acknowledged those messages.
     *
     * @return Store The publish store for this client.
     */
    public Store getPublishStore()
    {
        return this.publishStore;
    }

    /**
     * Connects to the AMPS instance through the provided URI.
     * The URI is a {@link String} with the format:
     *   <code>transport://host:port/protocol</code>
     * Notice that the protocol can be independent of the message type. 60East
     * recommends using the <code>amps</code> protocol, although some
     * installations use one of the legacy protocols such as <code>fix</code>,
     * <code>nvfix</code> or <code>xml</code>. Contact your server
     * administrator for the correct URI for the instance.
     *
     * @param uri The URI string to connect to
     * @throws ConnectionRefusedException The connection could not be established
     * @throws AlreadyConnectedException The connection is already connected
     * @throws InvalidURIException The specified URI is invalid
     * @throws ProtocolException The protocol is invalid
     * @throws TransportTypeException The transport type is invalid
     */
    public void connect(String uri) throws ConnectionException
    {
        lock.lock();
        try
        {
            try
            {
                this.uri = new URI(uri);
            }
            catch (URISyntaxException urisex)
            {
                throw new InvalidURIException(urisex);
            }
            if(this.transport == null)
            {
                Properties props = new URIProperties(this.uri);
                String sProtocol = this.uri.getPath().substring(1, this.uri.getPath().length());
                Protocol messageType = ProtocolFactory.createProtocol(sProtocol, props);
                if(version == Version.AMPS_1 && messageType instanceof FIXProtocol)
                {
                    ((FIXProtocol)messageType).setV1(true);
                }
                this.transport = TransportFactory
                                 .createTransport(this.uri.getScheme(), messageType, props);
                this.message = this.transport.allocateMessage();
                this.beatMessage = this.transport.allocateMessage();
                this.beatMessage.setCommand(Message.Command.Heartbeat);
                this.beatMessage.setOptions("beat");
                this.transport.setMessageHandler(this._handler);
                this.transport.setDisconnectHandler(this._handler);
            }
            this.transport.connect(this.uri);
            broadcastConnectionStateChanged(ConnectionStateListener.Connected);
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Sets the {@link ClientDisconnectHandler}. In the event that the Client is
     * disconnected from AMPS, the <code>invoke</code> method from the
     * ClientDisconnectHandler will be invoked.
     *
     * @param disconnectHandler_ The disconnect handler
     */
    public void setDisconnectHandler(ClientDisconnectHandler disconnectHandler_)
    {
        this.disconnectHandler = disconnectHandler_;
    }

    /**
     * Returns the current {@link ClientDisconnectHandler} set on self.
     *
     * @return The current disconnect handler.
     */
    public ClientDisconnectHandler getDisconnectHandler()
    {
        return this.disconnectHandler;
    }

    /**
     *
     * Sets the {@link MessageHandler} instance used when no other handler matches.
     *
     * @param messageHandler The message handler used when no other handler matches.
     *
     * @deprecated
     * Use {@link setLastChanceMessageHandler } instead.
     */
    public void setUnhandledMessageHandler(MessageHandler messageHandler)
    {
        setLastChanceMessageHandler(messageHandler);
    }

    /**
     * Sets the {@link MessageHandler} instance used when no other handler matches.
     *
     * @param messageHandler The message handler used when no other handler matches.
     */
    public void setLastChanceMessageHandler(MessageHandler messageHandler)
    {
        this.lastChanceMessageHandler = messageHandler;
    }
    /**
     * Sets the {@link java.beans.ExceptionListener} instance used for communicating absorbed exceptions.
     *
     * @param exceptionListener The exception listener instance to invoke for exceptions.
     */
    public void setExceptionListener(ExceptionListener exceptionListener)
    {
        this.exceptionListener = exceptionListener;
    }

    /**
     * Sets the {@link SubscriptionManager} instance used for recording active subscriptions.
     *
     * @param subscriptionManager The subscription manager invoked when the subscriptions change.
     */
    public void setSubscriptionManager(SubscriptionManager subscriptionManager)
    {
        this.subscriptionManager = subscriptionManager;
    }

    /**
     * Returns the {@link SubscriptionManager} instance used for recording active subscriptions.
     *
     * @return The current subscription manager.
     */
    public SubscriptionManager getSubscriptionManager()
    {
        return this.subscriptionManager;
    }
    private void absorbedException(Exception e)
    {
        try
        {
            if(this.exceptionListener != null)
            {
                this.exceptionListener.exceptionThrown(e);
            }
        }
        catch(Exception e2)
        {
            ; // Absorb exception from the exception listener
        }
    }

    /**
     * Sets the {@link MessageHandler} instance used for duplicate messages.
     *
     * @param messageHandler The message handler to invoke for duplicate messages
     */
    public void setDuplicateMessageHandler(MessageHandler messageHandler)
    {
        this.duplicateMessageHandler = messageHandler;
    }

    /**
     * Sets the {@link FailedWriteHandler} instance used to report on failed
     * messages that have been written.
     *
     * @param handler_  The handler to invoke for published duplicates.
     */
    public void setFailedWriteHandler(FailedWriteHandler handler_)
    {
        this._failedWriteHandler = handler_;
    }

    /**
     * Adds a {@link ConnectionStateListener} instance that will be invoked when this client connects or disconnects.
     * @param listener_ The instance to invoke.
     * @since 4.0.0.0
     */
    public void addConnectionStateListener(ConnectionStateListener listener_)
    {
        lock.lock();
        try
        {
            this._connectionStateListeners.add(listener_);
        }
        finally
        {
            lock.unlock();
        }
    }
    
    /**
     * Removes a {@link ConnectionStateListener} from being invoked when this client connects or disconnects.
     * @param listener_ The instance to remove.
     * @since 4.0.0.0
     */
    public void removeConnectionStateListener(ConnectionStateListener listener_)
    {
        lock.lock();
        try
        {
            this._connectionStateListeners.remove(listener_);
        }
        finally
        {
            lock.unlock();
        }
    }
    
    protected void broadcastConnectionStateChanged(int newState_)
    {
        lock.lock();
        for(ConnectionStateListener listener : _connectionStateListeners)
        {
            try
            {
                listener.connectionStateChanged(newState_);
            }
            catch(Throwable t)
            {
                // Discard these exceptions.
            }
        }
        lock.unlock();
    }
    /**
     * Disconnect from the AMPS server.
     *
     */
    public void disconnect()
    {
        Transport currentTransport = null;
        lock.lock();
        currentTransport = this.transport;
        heartbeatTimer.setTimeout(0);
        lock.unlock();

        // don't disconnect the transport with the client lock held.
        if(currentTransport != null) currentTransport.disconnect();

        lock.lock();
        try
        {
            // only disconnect if a transport has been created
            _routes.clear();
            // Signal and then clear the ack info
            cancelSynchronousWaiters(Integer.MAX_VALUE);
            broadcastConnectionStateChanged(ConnectionStateListener.Disconnected);
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
        finally
        {
            lock.unlock();
        }
    }


    /**
     * Disconnect from the AMPS server.
     *
     */
    public void close()
    {
        disconnect();
    }

    /**
     * Creates a new {@link Message} appropriate for this client. This function should be
     *   called rarely, since it does allocate a handful of small objects.  Users sensitive
     *   to garbage collection delays should cache the message object for later usage.
     *
     * @return A new {@link Message} instance
     */
    public Message allocateMessage()
    {
        try
        {
            lock.lock();
            return this.transport.allocateMessage();
        } finally {
            lock.unlock();
        }
    }

    /**
     * Send a {@link Message} to AMPS via the {@link Transport} used in the Client.
     * This method is provided for special cases. In general, you can get the
     * same results with additional error checking by using a {@link Command}
     * object with the execute or executeAsync methods.
     *
     * @param message The message to send
     * @throws DisconnectedException The connection was disconnected at time of send
     */
    public void send(Message message) throws DisconnectedException
    {
        lock.lock();
        try
        {
            sendInternal(message);
        }
        finally
        {
            lock.unlock();
        }
    }

    private int sendInternal(Message message) throws DisconnectedException
    {
        assert ((ReentrantLock)lock).isHeldByCurrentThread();
        while(true)
        {
            // this.transport is only safe to access while in the lock.
            Transport currentTransport = this.transport;
            int version = currentTransport.getVersion();
            try
            {
                currentTransport.sendWithoutRetry(message);
                return version;
            }
            catch(DisconnectedException d)
            {
                // attempt to reconnect.  If we get anything
                // other than a Retry(meaning it succeeded),
                // let the exception flow upwards.
                lock.unlock();
                try
                {
                    currentTransport.handleCloseEvent(version, "Error occured while sending message", d);
                }
                catch(RetryOperationException r)
                {
                    if (message.getCommand() == Message.Command.Subscribe) {
                        // The original message was a subscribe, but the successful call
                        // to handleCloseEvent() included a resubscription so there's
                        // no need to retry sending the original subscribe message
                        //
                        // For testing purposes, you can dramatically increase the
                        // frequency of this case by adding the following to the head of
                        // TCPTransportImpl.handleCloseEvent():
                        //
                        // if (Thread.currentThread().getName().indexOf("Reader Thread") != -1) {
                        //     return;
                        // }
                        //
                        return currentTransport.getVersion();
                    }
                }
                finally
                {
                    lock.lock();
                }
            }
        }
    }

    /**
     * Send a {@link Message} to AMPS and register the messageHandler for any messages
     * resulting from the command execution.
     *
     * This method is provided for special cases. In general, you can get the
     * same results with additional error checking by using a {@link Command}
     * object with the execute or executeAsync methods.
     *
     * @param messageHandler The message handler that'll receive messages for this command
     * @param message The message to send
     * @param timeout The number of milliseconds to wait for command acknowledgment
     * @return The command identifier assigned to this command, or null if none is assigned.
     * @throws AMPSException An exception occured while sending or waiting for a response to this Message.
     */
    @SuppressWarnings("fallthrough")
    public CommandId send(MessageHandler messageHandler, Message message, long timeout)
    throws AMPSException
    {
        CommandId id = null;
        String strId = message.getCommandId();
        if (strId != null)
        {
            id = new CommandId(strId);
        }
        int requestedAcks = message.getAckType();
        int systemAddedAcks = 0;
        boolean isSubscribe = false;
        switch(message.getCommand())
        {
            // Subscriptions need to have a sub id
        case Message.Command.Subscribe:
        case Message.Command.DeltaSubscribe:
            if (serverVersion >= MIN_PERSISTED_BOOKMARK_VERSION &&
                message.getBookmark() != null)
            {
                systemAddedAcks |= Message.AckType.Processed | Message.AckType.Persisted;
            }
            // fall through
        case Message.Command.SOWAndSubscribe:
        case Message.Command.SOWAndDeltaSubscribe:
            if (id == null)
            {
                id = CommandId.nextIdentifier();
                message.setCommandId(id);
            }
            if (message.getSubId() == null)
            {
                message.setSubId(id);
            }
            message.setSendMatchingIds(true);
            // SOW queries just need a QueryId.
            // fall through.
            isSubscribe = true;
        case Message.Command.SOW:
            if(id == null)
            {
                id = CommandId.nextIdentifier();
                message.setCommandId(id);
            }
            if (message.getQueryId() == null)
            {
                message.setQueryId(id);
            }
            systemAddedAcks |= Message.AckType.Processed;
            // for SOW only, we get a completed ack so we know when to remove the handler.
            if(!isSubscribe) systemAddedAcks |= Message.AckType.Completed;
            requestedAcks |= systemAddedAcks;
            message.setAckType(requestedAcks);
            _routes.addRoute(id, messageHandler, requestedAcks, systemAddedAcks, isSubscribe);
            lock.lock();
            try
            {
                this.syncAckProcessing(id, message, timeout);
            }
            catch(TimedOutException e)
            {
                _routes.removeRoute(id);
                throw e;
            }
            finally
            {
                lock.unlock();
            }
            break;
            // These are valid commands that are used as-is
        case Message.Command.Unsubscribe:
        case Message.Command.Heartbeat:
        case Message.Command.Logon:
        case Message.Command.StartTimer:
        case Message.Command.StopTimer:
        case Message.Command.DeltaPublish:
        case Message.Command.Publish:
        case Message.Command.SOWDelete:
            // if an ack is requested, it'll need a command ID.
            if(message.getAckType() != Message.AckType.None)
            {
                if (id == null)
                {
                    id = CommandId.nextIdentifier();
                    message.setCommandId(id);
                }
                if(messageHandler != null)
                {
			_routes.addRoute(id, messageHandler, message.getAckType(), 0, false);
                }
            }
            this.send(message);
            break;
            // These are things that shouldn't be sent (not meaningful)
        case Message.Command.GroupBegin:
        case Message.Command.GroupEnd:
        case Message.Command.OOF:
        case Message.Command.Ack:
        case Message.Command.Unknown:
        default:
            throw new CommandException("Command type can not be sent directly to AMPS");
        }
        return id;
    }

  /**
   *  Execute the provided command and return messages received in response
   *  in a {@link MessageStream}.
   *
   *  This method creates a message based on the provided {@link Command},
   *  sends the message to AMPS, and receives the results. AMPS sends the
   *  message and receives the results on a background thread. That thread
   *  populates the MessageStream returned by this method.
   *
   *  @param command The Command object containing the command to send to AMPS
   *  @return A MessageStream that provides messages received in response to the command
   * @since 4.0.0.0
   */

    public MessageStream execute(Command command) throws AMPSException
    {
        // If the command is sow and has a sub_id, OR
        // if the command has a replace option, return the existing
        // messagestream, don't create a new one.

        boolean useExistingHandler = command.getSubId() != null && ((command.getOptions() != null && command.getOptions().contains("replace")) || command.getCommand() == Message.Command.SOW);
        if(useExistingHandler)
        {
            // Try to find the exist message handler.
            byte[] subId = command.getSubId().id;
            if(subId != null)
            {
                MessageHandler existingHandler = _routes.findRoute(command.getSubId());
                if (existingHandler != null)
                {
                    // we found an existing handler. It might not be a message stream, but that's okay.
                    executeAsync(command, existingHandler);
                    // This will return null if handler was not a messagestream.
                    return existingHandler instanceof MessageStream ? (MessageStream)existingHandler:null;
                }
            }
            // fall through; we'll a new handler altogether.
        }

        MessageStream result = new MessageStream(this);
        CommandId cid = executeAsync(command, result);
        boolean statsAck = (command.getAckType() & Message.AckType.Stats) > 0;

        if(statsAck)
        {
            result.setStatsOnly();
        }
        else if (command.getCommand() == Message.Command.SOW)
        {
            result.setSOWOnly();
        }
        else if(cid!=null)
        {
            result.setSubscription(cid);
        }
        else { result = null; }
        
        return result;
    }

  /**
   *  Execute the provided command on a background thread and provide
   *  messages received in response to the handler provided.
   *
   *  This method creates a message based on the provided {@link Command},
   *  sends the message to AMPS, and invokes the provided
   *  {@link MessageHandler } to process messages returned in response
   *  to the command. AMPS sends the message and receives the results
   *  on a background thread. The MessageHandler runs on the background thread.
   *
   *  @param command The Command object containing the command to send to AMPS
   *  @param handler The MessageHandler to invoke to process messages received 
   *  @return the CommandId for the command
   * @since 4.0.0.0
   */
    
    public CommandId executeAsync(Command command, MessageHandler handler) throws AMPSException
    {
        lock.lock();
        try
        {
            return executeAsyncNoLock(command, handler);
        } finally
        {
            lock.unlock();
        }
    }

    // This is a bit complicated, but we're basically doing all the stuff we need to do here to get the Client's state
    // to reflect the command we're about to send -- for example, record the subscription, set a client sequence number,
    // add a handler, and determine whether or not to wait for a processed ack.
    private CommandId executeAsyncNoLock(Command command, MessageHandler messageHandler)
    throws AMPSException
    {
        CommandId cid = command.prepare(this);
        int systemAddedAcks = Message.AckType.Processed;
        boolean isPublishStore = this.publishStore!=null && command.needsSequenceNumber();
        if(command.getCommand() == Message.Command.SOW) systemAddedAcks |= Message.AckType.Completed;

        // fixup bookmark and apply ack types as necessary.
        if(command.getBookmark() != null)
        {
            if(command.isSubscribe())
            {
                if(serverVersion >= MIN_PERSISTED_BOOKMARK_VERSION)
                {
                    systemAddedAcks |= Message.AckType.Persisted;
                    command._message.setAckType(Message.AckType.Persisted|command.getAckType());
                }
            }
            if(command.getBookmark().equals(Client.Bookmarks.MOST_RECENT))
            {
                if (this.bookmarkStore != null)
                {
                    Field mostRecent = bookmarkStore.getMostRecent(command._message.getSubIdRaw());
                    command._message.setBookmark(mostRecent.buffer, mostRecent.position, mostRecent.length);
                }
                else
                    command._message.setBookmark(Bookmarks.EPOCH);
            }
        }
        // If this is destined for a publish store, make sure to request a persisted ack before we add routes
        if(isPublishStore)
        {
            systemAddedAcks |= Message.AckType.Persisted;
            command._message.setAckType(Message.AckType.Persisted|command.getAckType());
        }
        // register any message handler supplied.
        if(messageHandler!=null)
        {
            CommandId subid = command.getSubId();
            CommandId qid = command.getQueryId();
            boolean isSubscribe = command.isSubscribe();
            if(qid != null)
            {
                _routes.addRoute(qid, messageHandler, command.getAckType(), systemAddedAcks, isSubscribe);
            }
            if(subid!=null)
            {
                _routes.addRoute(subid, messageHandler, command.getAckType(), systemAddedAcks, isSubscribe);
            } else if (cid!=null)
            {
                _routes.addRoute(cid, messageHandler, command.getAckType(), systemAddedAcks, isSubscribe);
            } else
            {
                throw new IllegalArgumentException("To use a messagehandler, you must also supply a command or subscription ID.");
            }
        }
        boolean useSyncSend = cid!=null && (command.getAckType()&Message.AckType.Processed)>0;
        
        // if there's a command id & a processed ack type set, use syncAckProcessing.
        if(isPublishStore)
        {
            while (true)
            {
                // It's not safe to HA publish while a reconnect is in progress.
                // When a reconnect is in progress, we might be able to sneak
                // a publish through, but if it's sequence number is higher
                // than the beginning of what we need to replay, the server will
                // take that message to mean that we're done replaying, and ignore
                // what we replay.  If, once we have the lock, we discover
                // it's a bad time to HA publish, we'll spin waiting for the
                // reconnect to finish up, and then go.
                if(!_badTimeToHAPublish)
                {
                    long haSeqNumber = this.haSequenceNumber++;
                    command._message.setSequence(haSeqNumber);
                    command.setClientSequenceNumber(haSeqNumber);
                    byte[] data = null;
                    long dataOffset = 0;
                    long dataLength = 0;
                    byte[] corId = command._message._CorrelationId.buffer;
                    long corIdOffset = command._message._CorrelationId.position;
                    long corIdLength = command._message._CorrelationId.length;
                    
                    // The expiration will be the type of SOWDelete or the expiration
                    int expiration = -1;
                    // Publish, delta_publish, sow_delete by data
                    StringField topic = command._message._Topic;
                    if (command._message._Data.length != 0)
                    {
                        data = command._message._Data.buffer;
                        dataOffset = command._message._Data.position;
                        dataLength = command._message._Data.length;
                        if (command.Command != Message.Command.SOWDelete)
                        {
                            if (command.hasExpiration())
                            {
                                expiration = command.Expiration;
                                this.publishStore.store(haSeqNumber,
                                                        command.Command, topic.buffer,
                                                        topic.position,
                                                        topic.length,
                                                        data, dataOffset, dataLength,
                                                        corId, corIdOffset, corIdLength,
                                                        expiration);
                            }
                            else
                            {
                                this.publishStore.store(haSeqNumber,
                                                        command.Command, topic.buffer,
                                                        topic.position,
                                                        topic.length,
                                                        data, dataOffset, dataLength,
                                                        corId, corIdOffset, corIdLength);
                            }
                        }
                        else
                        {
                            expiration = Store.SOWDeleteByData;
                            this.publishStore.store(haSeqNumber,
                                                    command.Command, topic.buffer,
                                                    topic.position,
                                                    topic.length,
                                                    data, dataOffset, dataLength,
                                                    expiration, cid);
                        }
                        
                    }
                    else if (command.Filter != null) // sow_delete by filter
                    {
                        Field filter = command._message.getFilterRaw();
                        dataLength = filter.length;
                        data = filter.buffer;
                        dataOffset = filter.position;
                        expiration = Store.SOWDeleteByFilter;
                        this.publishStore.store(haSeqNumber,
                                                command.Command, topic.buffer,
                                                topic.position,
                                                topic.length,
                                                data, dataOffset, dataLength,
                                                expiration, cid);
                    }
                    else // sow_delete by keys
                    {
                        Field sowKeys = command._message.getSowKeysRaw();
                        dataLength = sowKeys.length;
                        data = sowKeys.buffer;
                        dataOffset = sowKeys.position;
                        expiration = Store.SOWDeleteByKeys;
                        this.publishStore.store(haSeqNumber,
                                                command.Command, topic.buffer,
                                                topic.position,
                                                topic.length,
                                                data, dataOffset, dataLength,
                                                expiration, cid);
                    }
                    if(useSyncSend) syncAckProcessing(cid,command._message,command.getTimeout()); else sendInternal(command._message);
                    break;
                } else {
                    lock.unlock();
                    try
                    {
                        Thread.yield();
                    }
                    catch (Exception e)
                    {
                    }
                    lock.lock();
                }
            }
        }
        else
        {
            if(useSyncSend) syncAckProcessing(cid,command._message,command.getTimeout()); else sendInternal(command._message);
        }
        if(command.isSubscribe())
        {
            // register with the subscription manager.
            this.subscriptionManager.subscribe(messageHandler, command.getCommand(), command.getSubId(), command.getTopic(), 
                command.getFilter(), command.getBookmark(), command.getOptions(), command.getBatchSize()); 
            return command.getSubId();
        }
        return cid;
    }
    /**
     * Requests a server heartbeat, and configures the client to close the connection
     * if a heartbeat (or other activity) is not seen on the connection.
     *
     * @param intervalSeconds_  The time (seconds) between beats from the server.
     * @param timeoutSeconds_ The time (seconds) to allow silence on the connection before assuming it is dead.
     * @throws DisconnectedException The client became disconnected while attempting to send the heartbeat request.
     */
    public void setHeartbeat(int intervalSeconds_, int timeoutSeconds_) throws DisconnectedException
    {
        if(this.transport != null)
        {
            lock.lock();
            try
            {
                if(intervalSeconds_ > 0)
                {
                    this.message.reset();
                    this.message.setCommand(Message.Command.Heartbeat);
                    this.message.setOptions(String.format("start,%d",intervalSeconds_));
                    this.heartbeatTimer.setTimeout(intervalSeconds_ * 1000);
                    this.heartbeatTimer.start();
                    sendInternal(this.message);
                }
                this.transport.setReadTimeout(timeoutSeconds_ * 1000);
            }
            finally
            {
                lock.unlock();
            }
        }
    }

    /**
     * Requests a server heartbeat, and configures the client to close the connection
     * if a heartbeat (or other activity) is not seen on the connection after two heartbeat intervals.
     *
     * @param intervalSeconds_  The time (seconds) between beats from the server.
     * @throws DisconnectedException The client became disconnected while attempting to send the heartbeat request.
     */
    public void setHeartbeat(int intervalSeconds_) throws DisconnectedException
    {
        setHeartbeat(intervalSeconds_, intervalSeconds_ * 2);
    }

    /**
     * Publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param  topic          the topic to publish to
     * @param  topicOffset    the offset in topic where the topic begins
     * @param  topicLength    the length of the topic in the topic array
     * @param  data           the data to publish
     * @param  dataOffset     the offset in data where data begins
     * @param  dataLength     the length of the data
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     * @return the sequence number for the message when a persistent store is configured for the client, 0 if no store is configured
     */
    public long publish(
        byte[] topic, int topicOffset, int topicLength,
        byte[] data , int dataOffset , int dataLength)
    throws AMPSException
    {
        if(topicLength ==0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.Publish).setTopic(topic,topicOffset,topicLength)
                .setData(data,dataOffset,dataLength);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
            return command.getClientSequenceNumber();
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param data Data to publish to the topic
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     * @return the sequence number for the message when a persistent store is configured for the client, 0 if no store is configured
     */
    public long publish(
        String topic,
        String data)
    throws AMPSException
    {
        if(topic==null || topic.length() ==0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.Publish).setTopic(topic).setData(data);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
            return command.getClientSequenceNumber();
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param  topic          the topic to publish to
     * @param  topicOffset    the offset in topic where the topic begins
     * @param  topicLength    the length of the topic in the topic array
     * @param  data           the data to publish
     * @param  dataOffset     the offset in data where data begins
     * @param  dataLength     the length of the data
     * @param  expiration     the number of seconds until the message expires
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     * @return the sequence number for the message when a persistent store is configured for the client, 0 if no store is configured
     */
    public long publish(
        byte[] topic, int topicOffset, int topicLength,
        byte[] data , int dataOffset , int dataLength,
        int expiration)
    throws AMPSException
    {
        if(topicLength == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.Publish).setTopic(topic,topicOffset,topicLength).setData(data,dataOffset,dataLength).setExpiration(expiration);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
            return command.getClientSequenceNumber();
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param data Data to publish to the topic
     * @param expiration the number of seconds until the message expires
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     * @return the sequence number for the message when a persistent store is configured for the client, 0 if no store is configured
     */
    public long publish(
        String topic,
        String data,
        int expiration)
    throws AMPSException
    {
        if(topic==null || topic.length() == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.Publish).setTopic(topic).setData(data).setExpiration(expiration);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
            return command.getClientSequenceNumber();
        }
        finally
        {
            lock.unlock();
        }
    }    
    /**
     * Delta publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param topicOffset offset into topic array where the topic name begins
     * @param topicLength length of topic array
     * @param data Data to publish to the topic
     * @param dataOffset offset into data array where the data begins
     * @param dataLength length of data array
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     */
    public void deltaPublish(
        byte[] topic, int topicOffset, int topicLength,
        byte[] data, int dataOffset, int dataLength)
    throws
        AMPSException
    {
        if(topicLength == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaPublish).setTopic(topic,topicOffset,topicLength)
                .setData(data,dataOffset,dataLength);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Delta publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param data Data to publish to the topic
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     */
    public void deltaPublish(
        String topic, String data)
    throws
        AMPSException
    {
        if(topic==null || topic.length() == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaPublish)
                                .setTopic(topic)
                                .setData(data);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command,null);
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Delta publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param topicOffset offset into topic array where the topic name begins
     * @param topicLength length of topic array
     * @param data Data to publish to the topic
     * @param dataOffset offset into data array where the data begins
     * @param dataLength length of data array
     * @param expiration the number of seconds until the message expires
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     */
    public void deltaPublish(
        byte[] topic, int topicOffset, int topicLength,
        byte[] data, int dataOffset, int dataLength, int expiration)
    throws
    AMPSException
    {
        if(topicLength == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaPublish).setTopic(topic,topicOffset,topicLength)
                .setData(data,dataOffset,dataLength).setExpiration(expiration);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Delta publish a message to an AMPS topic. If the client
     * was created with a persistent store on construction, then the client will store
     * before forwarding the message to AMPS.  In this store-and-forward case, the persistent
     * store will notify the user, when requested via callback, of the successful persistence
     * of the record within AMPS.
     *
     * @param topic Topic to publish the data to
     * @param data Data to publish to the topic
     * @param expiration the number of seconds until the message expires
     * @throws DisconnectedException The client was disconnected at time of publish
     * @throws StoreException An error occurred writing to the local HA store.
     */
    public void deltaPublish(
        String topic, String data, int expiration)
    throws
    AMPSException
    {
        if(topic==null || topic.length() == 0)
        {
          throw new InvalidTopicException("A topic must be specified.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaPublish).setTopic(topic).setData(data).setExpiration(expiration);
            if(this.publishStore != null) command.setAckType(Message.AckType.Persisted);
            executeAsyncNoLock(command, null);
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     *
     * Sends a start timer command to AMPS, which can be later stopped with
     * a stop timer command.
     *
     * @throws ConnectionException A connection error occurred while sending.
     */
    public void startTimer() throws AMPSException
    {
        lock.lock();
        try
        {
            executeAsyncNoLock(command.reset(Message.Command.StartTimer),null);
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     *
     * Sends a stop timer command to AMPS
     *
     * @return The command identifier.
     * @throws ConnectionException A connection error occured while sending.
     */
    public CommandId stopTimer(MessageHandler handler_) throws AMPSException
    {
        CommandId id = null;
        lock.lock();
        try
        {
            id = executeAsyncNoLock(command.reset(Message.Command.StopTimer).addAckType(Message.AckType.Stats),handler_);
        }
        finally
        {
            lock.unlock();
        }
        return id;
    }

    /**
     * Logs into AMPS with the parameters provided in the connect method.
     *
     * @param timeout The number of milliseconds to wait for the command to exceute.
     *
     * @return The command identifier.
     * @throws ConnectionException A connection error occured while logging on.
     * @throws StoreException An error occured on the local HA store.
     */
    public CommandId logon(long timeout) throws ConnectionException
    {
        return logon(timeout, new DefaultAuthenticator());
    }

    /**
     * Logs into AMPS with the parameters provided in the connect method.
     *
     * @param timeout The number of milliseconds to wait for the command to execute
     * @param authenticator The custom authenticator object to authenticate against
     * @return The command identifier
     * @throws TimedOutException The command execution exceeded the timeout value
     * @throws DisconnectedException The operation failed because the connection was invalid
     * @throws StoreException An error occured with the HA PublishStore.
     */
    public CommandId logon(long timeout, Authenticator authenticator)
    throws ConnectionException
    {
        if (authenticator == null)
        {
            authenticator = new DefaultAuthenticator();
        }

        CommandId id = CommandId.nextIdentifier();
        lock.lock();
        try
        {
            this.message.reset();
            this.message.setCommand(Message.Command.Logon);
            this.message.setCommandId(id);
            this.message.setClientName(this.name);
            if(this.uri.getUserInfo() != null)
            {
                // parse user info
                int upSep = this.uri.getUserInfo().indexOf(':');
                if(upSep < 0)
                {
                    this.username = this.uri.getUserInfo();
                    this.message.setUserId(this.username);
                    this.message.setPassword(authenticator.authenticate(this.username, null));
                }
                else
                {
                    this.username = this.uri.getUserInfo().substring(0,upSep);
                    String passToken = this.uri.getUserInfo().substring(upSep+1);
                    this.message.setUserId(this.username);
                    this.message.setPassword(
                        authenticator.authenticate(this.username, passToken));
                }

            }
            try
            {
                // Versions before 3.0 did not support the Processed ack
                //   for a logon command.
                if(this.version >= Version.AMPS_3)
                {
                    this.message.setAckType(Message.AckType.Processed);
                    String version = this.getVersion();
                    if (version != null && version.length() != 0) this.message.setVersion(version);
                    else this.message.setVersion("notinmanifest");
                    AckResponse response = null;
                    while(true)
                    {
                        response = this.syncAckProcessing(id, this.message, timeout);
                        if(response.state == Message.Status.Retry)
                        {
                            this.message.setPassword(authenticator.retry(response.username, response.password));
                            this.username = response.username;
                        }
                        else
                        {
                            break;
                        }
                    }
                    this.serverVersion = response.serverVersion;
                    if (bookmarkStore != null)
                    {
                        bookmarkStore.setServerVersion(this.serverVersion);
                    }
                    authenticator.completed(response.username, response.password, response.reason);
                }
                else
                {
                    sendInternal(this.message);
                }
            }
            catch(CommandException e)      // Absorb exceptions
            {
                this.absorbedException(e);
            }
            if(this.publishStore != null)
            {
                try
                {
                    this.publishStore.replay(new ClientStoreReplayer(this));
                }
                catch (StoreException storeException)
                {
                    throw new ConnectionException("A local store exception occured while logging on.",
                                                  storeException);
                }
            }
            return id;
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Logs into AMPS with the parameters provided in the connect method.
     *
     * @return The command identifier
     * @throws TimedOutException The command execution exceeded the timeout value
     * @throws DisconnectedException The operation failed because the connection was invalid
     * @throws StoreException An error occured writing to the local HA store.
     */
    public CommandId logon() throws ConnectionException
    {
        return logon(0);
    }


    /**
     * Places a bookmark subscription with AMPS.
     * Starts replay at the most recent message reported by the BookmarkStore.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param subId The subscription ID.  You may optionally provide a subscribtion ID
     *              to ease recovery scenarios, instead of having the system automatically
     *              generate one for you. When used with the 'replace' option, this
     *              is the subscription to be replaced. With a bookmark store,
     *              this is the subscription ID used for recovery. So, when
     *              using a persistent bookmark store, provide an explicit
     *              subscription ID that is consistent across application restarts.
     * @param bookmark The timestamp or bookmark location at which to start the subscription. This parameter can be the bookmark assigned by AMPS to a message, one of the special values in {@link Client.Bookmarks}, or a string of the format YYYYmmddTHHMMSS, as described in the AMPS User's guide. 
     * @param options A {@link Message.Options} value indicating desired options for this
     *        subscription.  Use Message.Options.None if no options are desired.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId bookmarkSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        CommandId subId,
        String bookmark,
        String options,
        long timeout)
    throws AMPSException
    {
        if(bookmark == null)
        {
            throw new AMPSException("A bookmark (or one of the Client.Bookmarks constants) is required to initiate a bookmark subscription.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.Subscribe).setTopic(topic).setFilter(filter).setOptions(options).setBookmark(bookmark);
            if(subId != null) command.setSubId(subId);
            executeAsyncNoLock(command, messageHandler);
            return command.getSubId();
        }
        finally
        {
            lock.unlock();
        }
    }
    
    /**
     * Places a bookmark delta subscription with AMPS.
     * Starts replay at the most recent message reported by the BookmarkStore.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param subId The subscription ID.  You may wish to supply your own subscription ID
     *        to ease recovery scenarios, instead of having the system generate one for you.
     * @param bookmark The timestamp or bookmark location at which to start the subscription. This parameter can be the bookmark assigned by AMPS to a message, one of the special values in {@link Client.Bookmarks}, or a string of the format YYYYmmddTHHMMSS, as described in the AMPS User's guide. 
     * @param options A {@link Message.Options} value indicating desired options for this
     *        subscription.  Use Message.Options.None if no options are desired.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId bookmarkDeltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        CommandId subId,
        String bookmark,
        String options,
        long timeout)
    throws AMPSException
    {
        if(bookmark == null)
        {
            throw new AMPSException("A bookmark (or one of the Client.Bookmarks constants) is required to initiate a bookmark delta subscription.");
        }
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaSubscribe).setTopic(topic).setFilter(filter).setOptions(options).setBookmark(bookmark);
            if(subId != null) command.setSubId(subId);
            executeAsyncNoLock(command, messageHandler);
            return command.getSubId();
        }
        finally
        {
            lock.unlock();
        }
    }
    
    /**
     * Places a subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId subscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        long timeout)
    throws AMPSException
    {
        return subscribe(messageHandler, topic, filter,
                Message.Options.None, timeout);
    }

    /**
     * Places a subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId subscribe(MessageHandler messageHandler,
            String topic,
            String filter,
            String options,
            long timeout) throws AMPSException
    {
        return subscribe(messageHandler, topic, filter,
                options, timeout, null);
    }

    /**
     * Places a subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @param subId The subscription id to use for the subscription.
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId subscribe(MessageHandler messageHandler,
            String topic,
            String filter,
            String options,
            long timeout,
            String subId) throws AMPSException
    {
        lock.lock();
        try
        {
            if (subId == null && options != null && options.contains("replace"))
                throw new CommandException("Cannot issue a replacement subscription; a valid subscription id is required.");

            command.reset(Message.Command.Subscribe).setTopic(topic).setFilter(filter).setOptions(options).setTimeout(timeout).setSubId(subId);
            executeAsyncNoLock(command, messageHandler);
            return command.getSubId();
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Places a subscription with AMPS.
     *
     * @param topic The topic to subscribe to
     * @return a MessageStream to iterate over.
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream subscribe(String topic)
    throws AMPSException
    {
        return execute(new Command("subscribe").setTopic(topic));
    }
    /**
     * Places a subscription with AMPS.
     *
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @return a MessageStream to iterate over.
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream subscribe(String topic, String filter)
    throws AMPSException
    {
        return execute(new Command("subscribe").setTopic(topic).setFilter(filter));
    }

    /**
     * Places a subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId subscribe(
        MessageHandler messageHandler,
        String topic,
        long timeout)
    throws AMPSException
    {
        return subscribe(messageHandler, topic, null, timeout);
    }

    /**
     * Places a delta subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId deltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        String options,
        long timeout)
    throws AMPSException
    {
        return deltaSubscribe(messageHandler, topic, filter, options, timeout, null);
    }

    /**
     * Places a delta subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @param subId The subscription id to use for the subscription.
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId deltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        String options,
        long timeout,
        String subId)
    throws AMPSException
    {
        lock.lock();
        try
        {
            command.reset(Message.Command.DeltaSubscribe).setTopic(topic).setFilter(filter).setOptions(options).setTimeout(timeout);
            if(subId != null) command.setSubId(new CommandId(subId));
            executeAsyncNoLock(command, messageHandler);
            return command.getSubId();
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Places a delta subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId deltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        long timeout)
    throws AMPSException
    {
        return deltaSubscribe(messageHandler, topic, filter, Message.Options.None, timeout);
    }
    /**
     * Places a delta subscription with AMPS.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId deltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        long timeout)
    throws AMPSException
    {
        return deltaSubscribe(messageHandler, topic, null, timeout);
    }

    /**
     * Remove a subscription from AMPS.
     *
     * @param subscriptionId The subscription identifier to remove
     * @throws DisconnectedException The client was disconnected at the time of execution
     */
    public void unsubscribe(CommandId subscriptionId) throws DisconnectedException
    {
        lock.lock();
        try
        {
            unsubscribeInternal(subscriptionId);
        }
        finally
        {
            lock.unlock();
        }
    }


    private void unsubscribeInternal(CommandId subscriptionId) throws DisconnectedException
    {
        CommandId id = CommandId.nextIdentifier();
        if(subscriptionId != null && _routes.removeRoute(subscriptionId))
        {
            this.message.reset();
            this.message.setCommand(Message.Command.Unsubscribe);
            this.message.setCommandId(id);
            this.message.setSubId(subscriptionId);
            this.subscriptionManager.unsubscribe(subscriptionId);
            sendInternal(this.message);
        }
    }
    /**
     * Remove all of the client's subscriptions from AMPS.
     *
     * @throws DisconnectedException The client was disconnected at the time of execution
     */
    public void unsubscribe() throws DisconnectedException
    {
        CommandId id = CommandId.nextIdentifier();
        lock.lock();
        try
        {
            this.message.reset();
            this.message.setCommand(Message.Command.Unsubscribe);
            this.message.setCommandId(id);
            this.message.setSubId("all");
            this.subscriptionManager.clear();
            sendInternal(this.message);
        }
        finally
        {
            lock.unlock();
        }
    }
    /**
     * Synchronously execute a SOW query.
     *
     * @param topic The topic to query
     * @param filter The filter
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream sow(String topic, String filter)
    throws AMPSException
    {
        return execute(new Command("sow").setTopic(topic).setFilter(filter).setBatchSize(10));
    }
    /**
     * Synchronously execute a SOW query.
     *
     * @param topic The topic to query
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream sow(String topic)
    throws AMPSException
    {
        return execute(new Command("sow").setTopic(topic).setBatchSize(10));
    }
    
    /**
     * Executes a SOW query.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param orderBy The ordering property
     * @param bookmark The timestamp or bookmark location for a historical query. This parameter can be the bookmark assigned by AMPS to a message, one of the special values in {@link Client.Bookmarks}, or a string of the format YYYYmmddTHHMMSS, as described in the AMPS User's guide. 
     * @param batchSize The batching parameter to use for the results
     * @param topN The maximum number of records the server will return (default is all that match)
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sow(MessageHandler messageHandler,
                         String topic,
                         String filter,
                         String orderBy,
                         String bookmark,
                         int batchSize,
                         int topN,
                         String options,
                         long timeout) throws AMPSException
    {
        lock.lock();
        try
        {
            executeAsyncNoLock(command.reset(Message.Command.SOW).setTopic(topic).setFilter(filter).setOrderBy(orderBy)
                    .setOptions(options).setBookmark(bookmark).setBatchSize(batchSize).setTopN(topN).setTimeout(timeout), messageHandler);
            return command.getCommandId();
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Executes a SOW query.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sow(MessageHandler messageHandler,
                         String topic,
                         String filter,
                         int batchSize,
                         String options,
                         long timeout) throws AMPSException
    {
        return sow(messageHandler,topic,filter,null,null,batchSize,-1,options,timeout);
    }

    /**
     * Executes a SOW query.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sow(MessageHandler messageHandler,
                         String topic,
                         String filter,
                         int batchSize,
                         long timeout) throws AMPSException
    {
        return sow(messageHandler, topic, filter, batchSize, Message.Options.None, timeout);
    }
    /**
     * Executes a SOW query.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param batchSize The batching parameter to use for the results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sow(MessageHandler messageHandler,
                         String topic,
                         int batchSize,
                         long timeout) throws AMPSException
    {
        return sow(messageHandler, topic, null, batchSize, timeout);
    }

    /**
     * Executes a SOW query.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sow(MessageHandler messageHandler,
                         String topic,
                         long timeout) throws AMPSException
    {
        // Default batchsize to 10 for good default performance
        return sow(messageHandler, topic, null, 10, timeout);
    }
    /**
     * Executes a SOW query and places a subscription.
     *
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @return A MessageStream to iterate over.
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream sowAndSubscribe(String topic, String filter)
    throws AMPSException
    {
        return execute(new Command("sow_and_subscribe").setTopic(topic).setFilter(filter).setBatchSize(10));
    }
    /**
     * Executes a SOW query and places a subscription.
     *
     * @param topic The topic to subscribe to
     * @return A MessageStream to iterate over.
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public MessageStream sowAndSubscribe(String topic)
    throws AMPSException
    {
        return execute(new Command("sow_and_subscribe").setTopic(topic).setBatchSize(10));
    }
    
    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param orderBy The ordering property
     * @param bookmark The timestamp or bookmark location for a historical query. This parameter can be the bookmark assigned by AMPS to a message, one of the special values in {@link Client.Bookmarks}, or a string of the format YYYYmmddTHHMMSS, as described in the AMPS User's guide. 
     * @param batchSize The batching parameter to use for the SOW query results
     * @param topN The maximum number of records the server will return (default is all that match)
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(MessageHandler messageHandler,
                                     String topic,
                                     String filter,
                                     String orderBy,
                                     String bookmark,
                                     int batchSize,
                                     int topN,
                                     String options,
                                     long timeout) throws AMPSException
     {
         lock.lock();
         try
         {
             executeAsyncNoLock(command.reset(Message.Command.SOWAndSubscribe).setTopic(topic).setFilter(filter).setOrderBy(orderBy)
                     .setOptions(options).setBookmark(bookmark).setBatchSize(batchSize).setTopN(topN).setTimeout(timeout), messageHandler);
             return command.getCommandId();
         }
         finally
         {
             lock.unlock();
         }
     }

    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(MessageHandler messageHandler,
                                     String topic,
                                     String filter,
                                     int batchSize,
                                     String options,
                                     long timeout) throws AMPSException
    {
      return sowAndSubscribe(messageHandler,topic,filter,null,null,batchSize,-1,options,timeout);
    }
    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param oofEnabled Specifies whether or not Out-of-Focus processing is enabled
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        int batchSize,
        boolean oofEnabled,
        long timeout) throws AMPSException
    {
        return sowAndSubscribe(messageHandler, topic, filter, null, null,batchSize,-1, oofEnabled?"oof":null, timeout);
    }
    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        int batchSize,
        long timeout) throws AMPSException
    {
        return sowAndSubscribe(messageHandler, topic, filter, batchSize, false, timeout);
    }


    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param batchSize The batching parameter to use for the SOW query results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(
        MessageHandler messageHandler,
        String topic,
        int batchSize,
        long timeout) throws AMPSException
    {
        return sowAndSubscribe(messageHandler, topic, null, batchSize, false, timeout);
    }

    /**
     * Executes a SOW query and places a subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndSubscribe(
        MessageHandler messageHandler,
        String topic,
        long timeout) throws AMPSException
    {
        return sowAndSubscribe(messageHandler, topic, null, 10, false, timeout);
    }

    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param orderBy The ordering property
     * @param batchSize The batching parameter to use for the SOW query results
     * @param topN The maximum number of records the server will return (default is all that match)
     * @param options A value from Message.Options indicating additional processing options
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(MessageHandler messageHandler,
                                          String topic,
                                          String filter,
                                          String orderBy,
                                          int batchSize,
                                          int topN,
                                          String options,
                                          long timeout)
      throws AMPSException
      {
          lock.lock();
          try
          {
              executeAsyncNoLock(command.reset(Message.Command.SOWAndDeltaSubscribe).setTopic(topic).setFilter(filter).setOrderBy(orderBy)
                      .setOptions(options).setBatchSize(batchSize).setTopN(topN).setTimeout(timeout), messageHandler);
              return command.getCommandId();
          }
          finally
          {
              lock.unlock();
          }
      }

    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param oofEnabled Specifies whether or not Out-of-Focus processing is enabled
     * @param sendEmpties Specifies whether or not unchanged records are received on the delta subscription
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(MessageHandler messageHandler,
                                          String topic,
                                          String filter,
                                          int batchSize,
                                          boolean oofEnabled,
                                          boolean sendEmpties,
                                          String options,
                                          long timeout)
    throws AMPSException
    {
      String opts = options;
      if (oofEnabled) 
      {
          opts = opts == null? "oof": opts+",oof";
      }
      
      if (!sendEmpties) 
      {
          opts = opts == null?"no_empties" : opts+",no_empties";
      }
      return sowAndDeltaSubscribe(messageHandler,topic,filter,null,batchSize,-1,opts,timeout);
    }
    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param oofEnabled Specifies whether or not Out-of-Focus processing is enabled
     * @param sendEmpties Specifies whether or not unchanged records are received on the delta subscription
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        int batchSize,
        boolean oofEnabled,
        boolean sendEmpties,
        long timeout) throws AMPSException
    {
        return sowAndDeltaSubscribe(messageHandler, topic, filter, batchSize, oofEnabled,
                sendEmpties, null, timeout);
    }
    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param batchSize The batching parameter to use for the SOW query results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        String filter,
        int batchSize,
        long timeout)
    throws AMPSException
    {
        return sowAndDeltaSubscribe(messageHandler, topic, filter, batchSize, false, true, timeout);
    }

    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param batchSize The batching parameter to use for the SOW query results
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        int batchSize,
        long timeout)
    throws AMPSException
    {
        return sowAndDeltaSubscribe(messageHandler, topic, null, batchSize, false, true, timeout);
    }

    /**
     * Executes a SOW query and places a delta subscription.
     *
     * @param messageHandler The message handler to invoke with matching messages
     * @param topic The topic to subscribe to
     * @param timeout The maximum time to wait for the subscription to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws SubscriptionAlreadyExistsException A subscription with this identifier is already registered
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowAndDeltaSubscribe(
        MessageHandler messageHandler,
        String topic,
        long timeout)
    throws AMPSException
    {
        return sowAndDeltaSubscribe(messageHandler, topic, null, 10, false, true, timeout);
    }

    /**
     * Executes a SOW delete with filter.
     *
     * @param messageHandler The message handler to invoke with stats and completed acknowledgements
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param options A value from Message.Options indicating additional processing options.
     * @param timeout The maximum time to wait for the SOW delete to be started to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowDelete(
        MessageHandler messageHandler,
        String topic,
        String filter,
        String options,
        long timeout)
    throws AMPSException
    {
        lock.lock();
        try
        {
            executeAsyncNoLock(command.reset(Message.Command.SOWDelete).setTopic(topic).setFilter(filter).addAckType(Message.AckType.Stats)
                    .setOptions(options).setTimeout(timeout), messageHandler);
            return command.getCommandId();
        }
        finally
        {
            lock.unlock();
        }
    
    }
    /**
     * Executes a SOW delete with filter.
     *
     * @param messageHandler The message handler to invoke with stats and completed acknowledgements
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param timeout The maximum time to wait for the SOW delete to be started to be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowDelete(
        MessageHandler messageHandler,
        String topic,
        String filter,
        long timeout) throws AMPSException
    {
        return this.sowDelete(messageHandler, topic, filter, null, timeout );
    }

    /**
     * Executes a SOW delete with filter.
     *
     * @param topic The topic to subscribe to
     * @param filter The filter
     * @param timeout The maximum time to wait for the SOW delete to be started to be placed (milliseconds)
     * @return The stats message returned by this delete.
     * @throws BadFilterException The provided filter is invalid
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public Message sowDelete(String topic,
        String filter,
        long timeout) throws AMPSException
    {
        lock.lock();
        try
        {
            return execute(command.reset(Message.Command.SOWDelete).addAckType(Message.AckType.Stats).setTopic(topic).setFilter(filter)
                    .setTimeout(timeout)).next();
        } finally {
            lock.unlock();
        }
    }
    /**
     * Executes a SOW delete using the SowKey assigned by AMPS to specify
     * the messages to delete. When you have the SowKeys for the messages,
     * this method is more efficient than using a filter to
     * delete messages from the SOW.
     *
     * For example, to efficiently delete a message that your program has
     * received from AMPS:
     * <pre><code>
     *  // DeleteMessageHandler receives a message containing
     *  // the results of the delete.
     *  DeleteMessageHandler dmh = new DeleteMessageHandler();
     *
     *  client.sowDeleteByKeys(dmh, message.getTopic(), message.getSowKey(), 1000); 
     * </code></pre>
     *
     * @param messageHandler The message handler to invoke with stats and completed acknowledgements
     * @param topic The topic to execute the SOW delete against
     * @param keys A comma separated list of SOW keys to be deleted
     * @param timeout The maximum time to wait for the SOW delete to be be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws BadRegexTopicException The topic specified was an invalid regular expression
     * @throws TimedOutException The operation took longer than the timeout to execute
     * @throws DisconnectedException The client wasn't connected when the operation was executed
     */
    public CommandId sowDeleteByKeys(
            MessageHandler messageHandler,
            String topic,
            String keys,
            long timeout) throws AMPSException
    {
        lock.lock();
        try
        {
            executeAsyncNoLock(command.reset(Message.Command.SOWDelete).setTopic(topic).setSOWKeys(keys).addAckType(Message.AckType.Stats).
                    setTimeout(timeout), messageHandler);
            return command.getCommandId();
        }
        finally
        {
            lock.unlock();
        }
    }
    
    /**
     * Executes a SOW delete by data. AMPS uses key fields in the data to
     * find and delete a message with the same keys.
     *
     * @param messageHandler The message handler to invoke with stats and completed acknowledgements
     * @param topic The topic to execute the SOW delete against
     * @param data The message to match and delete in the SOW cache
     * @param timeout The maximum time to wait for the SOW delete to be be placed (milliseconds)
     * @return The command identifier assigned to this command
     * @throws AMPSException An error occurred while waiting for this command to be processed.
     */
    public CommandId sowDeleteByData(
        MessageHandler messageHandler,
        String topic,
        String data,
        long timeout) throws AMPSException
    {
        lock.lock();
        try
        {
            return executeAsyncNoLock(command.reset(Message.Command.SOWDelete).addAckType(Message.AckType.Stats).
                    setTopic(topic).setData(data).setTimeout(timeout), messageHandler);
        }
        finally
        {
            lock.unlock();
        }
    }

    /** Clear the queued messages which may be waiting in the transport,
     * publish store or in the AMPS publish queue.  Method returns when
     * published messages have been flushed.
     *
     */
    public void publishFlush() throws DisconnectedException, TimedOutException
    {
        if (publishStore == null)
        {
            try
            {
                lock.lock();
                this.message.reset();
                CommandId id = CommandId.nextIdentifier();
                this.message.setCommandId(id);
                this.message.setAckType(Message.AckType.Processed);
                if (this.serverVersion >= MIN_FLUSH_VERSION)
                {
                    this.message.setCommand(Message.Command.Flush);
                }
                else
                {
                    this.message.setCommand(Message.Command.Publish);
                    this.message.setTopic("/AMPS/devnull");
                }
                this.syncAckProcessing(id, message, 0);
                return;
            }
            catch(NullPointerException e)
            {
                throw new DisconnectedException("not connected");
            }
            catch(AMPSException e)
            { ; }
            finally
            {
                lock.unlock();
            }
        }
        else
        {
            publishStore.flush();
        }
    }

    /**
     * Return when all published messages have been confirmed to have made it
     * through the publish queue inside the AMPS instnace.
     *
     * @param timeout Number of milliseconds to wait for flush
     *
     */
    public void publishFlush(long timeout) throws DisconnectedException,
           TimedOutException
    {
        if (publishStore == null)
        {
            try
            {
                lock.lock();
                this.message.reset();
                CommandId id = CommandId.nextIdentifier();
                this.message.setCommandId(id);
                this.message.setAckType(Message.AckType.Processed);
                if (this.serverVersion >= MIN_FLUSH_VERSION)
                {
                    this.message.setCommand(Message.Command.Flush);
                }
                else
                {
                    this.message.setCommand(Message.Command.Publish);
                    this.message.setTopic("/AMPS/devnull");
                }
                this.syncAckProcessing(id, message, timeout);
                return;
            }
            catch(NullPointerException e)
            {
                throw new DisconnectedException("not connected");
            }
            catch(AMPSException e)
            { ; }
            finally
            {
                lock.unlock();
            }
        }
        else
        {
            publishStore.flush(timeout);
        }
    }

    /**
     * Clear the queued messages which may be waiting in the transport
     *
     * @return long Number of messages or bytes queued after flush is completed or timed out
     *
     */
    public long flush() throws DisconnectedException
    {
        try
        {
            lock.lock();
            return this.transport.flush();
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Clear the queued messages which may be waiting in the transport
     *
     * @param timeout Number of milliseconds to wait for flush
     * @return long Number of messages or bytes queued after flush is completed or timed out
     *
     */
    public long flush(long timeout) throws DisconnectedException
    {
        try
        {
            lock.lock();
            return this.transport.flush(timeout);
        }
        catch(NullPointerException e)
        {
            throw new DisconnectedException("not connected");
        }
        finally
        {
            lock.unlock();
        }
    }

    /**
     * Return the build number for the client that is stored in the Manifes.mf of the jar file.
     *
     * @return String Build version number.
    */
    public String getVersion()
    {
        Manifest manifest;

        if (_version != null) return _version;
        synchronized(this)
        {
            _version = new String("unknown");

            // Get the class loader and all jar manifests
            ClassLoader cl = (ClassLoader) getClass().getClassLoader();
            if (cl == null)
            {
                return _version;
            }
            Enumeration<URL>  urls;
            try
            {
                urls = cl.getResources("META-INF/MANIFEST.MF");
            }
            catch (IOException E)
            {
                return _version;
            }
            if(urls == null)
            {
                return _version;
            }
            URL url = null;
            // Search for the amps_client.jar manifest and retrieve Release-Version
            while (urls.hasMoreElements())
            {
                url = urls.nextElement();
                if (url.toString().contains("amps_client.jar"))
                {
                    try
                    {
                        manifest = new Manifest(url.openStream());
                    }
                    catch (IOException E)
                    {
                        return _version;
                    }
                    Attributes attrs = manifest.getMainAttributes();
                    _version = (String)(attrs.getValue("Release-Version"));
                    break;
                }
            }
            return _version;
        }
    }

    /**
     * Assembles a new ConnectionInfo with the state of this client and
     * associated classes.
     * @return A new ConnectionInfo object.
     */
    public ConnectionInfo getConnectionInfo()
    {
        ConnectionInfo ci = new ConnectionInfo();
        ci.put("client.uri", this.uri == null ? null : this.uri.toString());
        ci.put("client.name", this.name);
        ci.put("client.username", this.username);
        if(this.publishStore != null)
        {
            ci.put("publishStore.unpersistedCount", this.publishStore.unpersistedCount());
        }
        return ci;
    }

    /**
     * This is a helper function used to encapsulate the logic for waiting on a
     * sent message so that async commands like <code>subscribe</code> and
     * <code>sow</code> can have an
     * easier time using an exception interface for failed command execution.
     *
     * @param m  Message to be sent
     * @param timeout Timeout in milliseconds
     * @throws DisconnectedException Connection was disconnected when trying to send
     * @throws BadFilterException Bad filter specified in query or subscription
     * @throws BadRegexTopicException Invalid regex topic specified in query or subscription
     * @throws SubscriptionAlreadyExistsException Subscription with the specified id already exists
     * @throws TimedOutException Operation took too long to execute
     * @throws NameInUseException Client name is already in use
     * @throws RetryOperationException Authentication module requested a retry
     * @throws AuthenticationException Client was unable to be authenticated
     * @throws NotEntitledException Client not entitled to that command or topic
     */
    private AckResponse syncAckProcessing(CommandId id, Message m, long timeout)
    throws
        DisconnectedException,
        BadFilterException,
        BadRegexTopicException,
        InvalidTopicException,
        SubscriptionAlreadyExistsException,
        TimedOutException,
        NameInUseException,
        RetryOperationException,
        AuthenticationException,
        NotEntitledException,
        SubidInUseException,
        CommandException,
        UnknownException
    {
        // NOTE: This is called with lock already held
        // Make an empty ack response and stick it in the map.
        // The reader thread takes care of setting it, removing it from the map,
        // and signaling everyone when an ack comes in.
        AckResponse ackResponse = new AckResponse();
        ackResponse.responded = false;
        _acks.put(id, ackResponse);
        ackResponse.connectionVersion = sendInternal(m);
        try
        {
            long startTime = System.currentTimeMillis();
            long now = startTime;
            while(!ackResponse.abandoned && !ackResponse.responded &&
                    (timeout == 0 || (now-startTime) <= timeout))
            {
                if(timeout > 0)
                {
                    long remainingTime = timeout - (now-startTime);
                    if(remainingTime > 0)
                    {
                        ackReceived.await(remainingTime, TimeUnit.MILLISECONDS);
                    }
                }
                else
                {
                    ackReceived.await();
                }
                now = System.currentTimeMillis();
            }
        }
        catch(InterruptedException ie)
        {
            ;  // It's OK to be interrupted, let's just continue on...
        }
        if(ackResponse.responded)
        {
            // We found the ack in the callback, therefore we have
            //  the status and failure reason when they're set.
            if(ackResponse.state != Message.Status.Failure)
            {
                if(m.getCommand() == Message.Command.Logon)
                {
                    // if we're a 'logon' command we need to extract the seq number,
                    // if available
                    if(this.publishStore != null)
                    {
                        try
                        {
                            this.publishStore.discardUpTo(ackResponse.sequence);
                        }
                        catch (AMPSException e)
                        {
                            ; // pass
                        }
                        // On a fresh client, we need to adopt the sequence
                        // number the server has for us.
                        if (this.haSequenceNumber < ackResponse.sequence + 1)
                        {
                            this.haSequenceNumber = ackResponse.sequence + 1;
                        }
                    }
                }
                // all is good, return
                return ackResponse;
            }
            switch(ackResponse.reason)
            {
            case Message.Reason.BadFilter:
                throw new BadFilterException("Filter '" + m.getFilter() + "' is invalid.");
            case Message.Reason.BadRegexTopic:
                throw new BadRegexTopicException("Regular Expression Topic '" + m.getTopic()
                                                 + "' is invalid.");
            case Message.Reason.InvalidTopic:
                throw new InvalidTopicException();
            case Message.Reason.SubscriptionAlreadyExists:
                throw new SubscriptionAlreadyExistsException("Subscription for command '"
                        + m.getCommandId() + "' already exists.");
            case Message.Reason.SubidInUse:
                throw new SubidInUseException("Subscription with subscription id '"
                        + m.getSubId() + "' already exists.");
            case Message.Reason.NameInUse:
                throw new NameInUseException("Client name \"" + m.getClientName() + "\" is already in use.");
            case Message.Reason.AuthFailure:
                throw new AuthenticationException("Logon failed for user \"" + m.getUserId() + "\"");
            case Message.Reason.NotEntitled:
                throw new NotEntitledException("User \"" + this.username + "\" not entitled to topic \"" + m.getTopic() + "\".");
            case Message.Reason.Other:
            case Message.Reason.None:
                throw new CommandException("Error from server: " + ackResponse.reasonText);
            }
        }
        else
        {
            // No ack found, must've timed out or disconnected waiting.
            if(!ackResponse.abandoned)
            {
                throw new TimedOutException("timed out waiting for operation");
            }
            else
            {
                throw new DisconnectedException("Connection closed while waiting for operation.");
            }
        }
        return ackResponse;
    }

    class ClientHandler implements MessageHandler, TransportDisconnectHandler
    {
        Client client = null;
        CommandId key = new CommandId();
        ClientHandler(Client client)
        {
            this.client = client;
        }

        public void preInvoke(int connectionVersion)
        {
            client.cancelSynchronousWaiters(connectionVersion);
        }
        /**
         * The MessageHandler implementation
         */
        public void invoke(Message m)
        {
            final int SOWMask = Message.Command.SOW | Message.Command.GroupBegin | Message.Command.GroupEnd;
            final int PublishMask = Message.Command.OOF | Message.Command.Publish | Message.Command.DeltaPublish;
            try
            {
                final int commandType = m.getCommand();

                if ((commandType & SOWMask) > 0)
                {
                    m.getQueryId(key);
                    _routes.deliverData(m, key);
                }
                else if ((commandType & PublishMask) > 0)
                {
                    Field subIds = m.getSubIdsRaw();
                    BookmarkField bookmark = m.getBookmarkRaw();
                    // Publish messages coming through on a subscription
                    int index = 0;
                    while( index < subIds.length )
                    {
                        int end = index;
                        for (; end < subIds.length && subIds.buffer[subIds.position + end] != (byte)','; ++end) ;
                        key.set(subIds.buffer, subIds.position + index, end-index);
                        index = end + 1;

                        MessageHandler handler = _routes.findRoute(key);
                        if(handler != null)
                        {
                            m.setSubId(key);
                            if(!bookmark.isNull())
                            {
                                // send it on to the BookmarkStore for logging,
                                // only log those messages we're really delivering.
                                // check to see if the log says its a duplicate
                                if(bookmarkStore.isDiscarded(m))
                                {
                                    // yes it is: don't send it
                                    try { this.client.duplicateMessageHandler.invoke(m); } catch (Exception e) { absorbedException(e); }
                                }
                                else  // not a dupe, log it.
                                {
                                    bookmarkStore.log(m);
                                    try { handler.invoke(m); } catch (Exception e) { absorbedException(e); }
                                }
                            } else {
                                try { handler.invoke(m); } catch (Exception e) { absorbedException(e); }
                            } // if !bookmark.isNull()
                        }// if handler!=null
                    } // while index < subIds.length
                }
                else if (commandType == Message.Command.Ack)
                {
                    int ackType = m.getAckType();

                    // call any handlers registered for this ack
                    _routes.deliverAck(m, ackType);
                    switch(ackType)
                    {
                    // Persisted acks are handled internally.
                    case Message.AckType.Persisted:
                        persistedAck(m);
                        return;
                        // Processed acks are used to respond to waiting synchonous commands.
                    case Message.AckType.Processed:
                        processedAck(m);
                        return;
                    default:
                        break;
                    }
                } else if (m.getCommand() == Message.Command.Heartbeat)
                {
                    checkAndSendHeartbeat(true);
                    return;
                }
                else if (_routes.deliverData(m) == 0)
                {
                    lastChanceMessageHandler.invoke(m);
                }
                checkAndSendHeartbeat(false);
            }
            catch(Exception e)
            {
                this.client.absorbedException(e);
            }
        }

        private void checkAndSendHeartbeat(boolean force)
        {
            if (force || client.heartbeatTimer.check())
            {
                try
                {
                    client.heartbeatTimer.start();
                    client.transport.sendWithoutRetry(client.beatMessage);
                } catch(Exception e)
                {
                    // we'll pass on handling the disconnect here
                }
            }
        }

        private void processedAck(Message message)
        {
            if(message.getCommandId(key))
            {
                // If this command is awaiting a Processed ack, let's notify it
                // that it's arrived
                lock.lock();
                try
                {
                    AckResponse response = null;
                    response = client._acks.remove(key);
                    if (response != null)
                    {
                        response.state = message.getStatus();
                        response.reason = message.getReason();
                        response.reasonText = message.getReasonText();
                        response.username = message.getUserId();
                        response.password = message.getPassword();
                        response.serverVersion = message.getVersionAsInt();
                        response.sequence = message.getSequenceRaw().isNull()?0:message.getSequence();
                        response.responded = true;  // Reset after setting Status/Reason
                        ackReceived.signalAll();
                        return;
                    }
                }
                catch (Exception e)
                {
                    this.client.absorbedException(e);
                }
                finally
                {
                    lock.unlock();
                }

                try
                {
                    lastChanceMessageHandler.invoke(message);
                }
                catch (Exception e)
                {
                    this.client.absorbedException(e);
                }
                return;
            }
        }

        private void persistedAck(Message message)
        {
            /*
             * Best Practice:  If you don't care about the dupe acks that occur during
             * failover or rapid disconnect/reconnect, then just ignore them.
             * We could discard each duplicate from the persisted store, but the
             * storage costs of doing 1 record discards is heavy.  In most scenarios
             * we'll just quickly blow through the duplicates and get back to
             * processing the non-dupes.
             */
            boolean handled=false;
            if (publishStore != null)
            {
                int reason = message.getReason();
                if(reason == Message.Reason.Duplicate || reason == Message.Reason.NotEntitled)
                {
                    if(client._failedWriteHandler != null)
                    {
                        try
                        {
                            long sequence = message.getSequence();
                            FailedWriteStoreReplayer replayer = new FailedWriteStoreReplayer(reason);
                            publishStore.replaySingle(replayer, sequence);
                            if(replayer.replayCount() > 0) handled = true;
                        }
                        catch (Exception e)
                        {
                            this.client.absorbedException(e);
                        }
                    }
                }
                else if(message.getStatus() == Message.Status.Success)
                {
                    try
                    {
                        LongField  sequence = (LongField)message.getSequenceRaw();
                        if (sequence != null && !sequence.isNull())
                        {
                            handled = true;
                            publishStore.discardUpTo(sequence.getValue());
                        }
                    }
                    catch (Exception e)
                    {
                        this.client.absorbedException(e);
                    }
                }

            }
            if (serverVersion >= MIN_PERSISTED_BOOKMARK_VERSION &&
                    bookmarkStore != null && !handled)
            {
                BookmarkField bookmark = message.getBookmarkRaw();
                if (bookmark != null && !bookmark.isNull() && bookmark.length>1)
                {
                    try
                    {
                        handled = true;
                        bookmarkStore.persisted(message.getSubIdRaw(), bookmark);
                    }
                    catch (AMPSException e)
                    {
                        this.client.absorbedException(e);
                    }
                }
            }
            if(!handled)
            {
              try
              {
                  lastChanceMessageHandler.invoke(message);
              }
              catch(Exception e)
              {
                  this.client.absorbedException(e);
              }
            }
        }

        // This is our way of "spying" on the activity of the user's disconnect handler.
        // We need to know if the user successfully connected to the underlying at least once,
        // to tell if they've "given up" on reconnecting or not.
        class TransportConnectionMeasurer implements Transport
        {
            Transport _t;
            boolean _successfullyConnected = false;
            public TransportConnectionMeasurer(Transport underlying)
            {
                _t = underlying;
            }

            public void clear()
            {
            	_successfullyConnected = false;
            }
            public boolean successfullyConnected()
            {
                return _successfullyConnected;
            }

            public void connect(URI uri)throws ConnectionRefusedException, AlreadyConnectedException, InvalidURIException
            {
                _t.connect(uri);
                _successfullyConnected = true;
            }

            public void close()
            {
                _t.close();
            }

            public void disconnect()
            {
                _t.disconnect();
            }

            public void setMessageHandler(MessageHandler ml)
            {
                _t.setMessageHandler(ml);
            }

            public void setDisconnectHandler(TransportDisconnectHandler dh)
            {
                _t.setDisconnectHandler(dh);
            }

            public void setExceptionListener(ExceptionListener exceptionListener)
            {
                _t.setExceptionListener(exceptionListener);
            }

            public void send(Message message) throws DisconnectedException
            {
                _t.send(message);
            }

            public void sendWithoutRetry(Message message) throws DisconnectedException
            {
                _t.sendWithoutRetry(message);
            }

            public Message allocateMessage()
            {
                return _t.allocateMessage();
            }

            public long writeQueueSize() throws DisconnectedException
            {
                return _t.writeQueueSize();
            }

            public long readQueueSize() throws DisconnectedException
            {
                return _t.readQueueSize();
            }

            public long flush() throws DisconnectedException
            {
                return _t.flush();
            }

            public long flush(long timeout) throws DisconnectedException
            {
                return _t.flush(timeout);
            }

            public void handleCloseEvent(int failedVersion_, String message, Exception e) throws DisconnectedException, RetryOperationException
            {
                _t.handleCloseEvent(failedVersion_, message, e);
            }

            public int getVersion()
            {
                return _t.getVersion();
            }
            public void setReadTimeout(int readTimeoutMillis_)
            {
                _t.setReadTimeout(readTimeoutMillis_);
            }
            public void setTransportFilter(TransportFilter tracer_)
            {
                _t.setTransportFilter(tracer_);
            }
        }
        /**
         * The TransportDisconnectHandler implementation
         */
        public void invoke(Transport newTransport, Exception e_)
        {
            Transport oldTransport = null;
            try
            {
                lock.lock();
                broadcastConnectionStateChanged(ConnectionStateListener.Disconnected);
                oldTransport = transport;
                TransportConnectionMeasurer measurer = new TransportConnectionMeasurer(newTransport);
                transport = measurer;
                while(true)
                {
                    try
                    {
                        _badTimeToHAPublish = true;
                        measurer.clear();
                        if (disconnectHandler instanceof ClientDisconnectHandler2)
                        {
                            ((ClientDisconnectHandler2)disconnectHandler).invoke(client, e_);
                        }
                        else
                        {
                            disconnectHandler.invoke(client);
                        }
                    }
                    finally
                    {
                        _badTimeToHAPublish = false;
                    }
                    // If the disconnect handler successfully connected to something,
                    // we keep going, even if the connection isn't alive now.
                    if(!measurer.successfullyConnected())
                    {
                        this.client.absorbedException(new DisconnectedException("reconnect failed."));
                        return;
                    }

                    try
                    {
                        this.client.subscriptionManager.resubscribe(client);
                        break;
                    } catch (TimedOutException ex)
                    {
                        this.client.absorbedException(ex);
                        // timed out attempting to resubscribe.
                        // pass
                    } catch (DisconnectedException ex)
                    {
                        this.client.absorbedException(ex);
                        // disconnected exception while attempting to resubscribe.
                        // pass.
                    }
                }
            }
            catch(Exception e)
            {
                this.client.absorbedException(e);
            }
            finally
            {
                transport = oldTransport;
                lock.unlock();
            }
        }

    }


    /**
     * Implementation of {@link StoreReplayer} to replay messages which were
     * recorded as published, but did not receive an ack. This is performed as
     * part of the logon function by default.
     *
     */
    class ClientStoreReplayer implements StoreReplayer
    {
        Client client;
        public ClientStoreReplayer(Client c)
        {
            this.client = c;
        }
        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLen,
                            byte[] corId, long corIdOff, long corIdLen) throws DisconnectedException
        {
            try
            {
                client.lock.lock();
                client.message.reset();
                client.message.setCommand(operation);
                client.message.getTopicRaw().set(topic, (int)topicOffset, (int)topicLen);
                client.message.getDataRaw().set(data, (int)dataOffset, (int)dataLen);
                client.message.getCorrelationIdRaw().set(corId,(int)corIdOff,(int)corIdLen);
                client.message.setAckType(Message.AckType.Persisted);
                client.message.setSequence(index);
                client.haSequenceNumber = index >= client.haSequenceNumber ? index+1 : client.haSequenceNumber;
                client.transport.sendWithoutRetry(message);
            }

            finally
            {
                client.lock.unlock();
            }
        }
        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLen, 
                            byte[] corId, long corIdOff, long corIdLen,int expiration) throws DisconnectedException
        {
            try
            {
                client.lock.lock();
                client.message.reset();
                client.message.setCommand(operation);
                client.message.getTopicRaw().set(topic, (int)topicOffset, (int)topicLen);
                client.message.getDataRaw().set(data, (int)dataOffset, (int)dataLen);
                client.message.getCorrelationIdRaw().set(corId,(int)corIdOff,(int)corIdLen);
                client.message.setExpiration(expiration);
                client.message.setSequence(index);
                client.message.setAckType(Message.AckType.Persisted);
                client.message.setSequence(index);
                client.haSequenceNumber = index >= client.haSequenceNumber ? index+1 : client.haSequenceNumber;
                client.transport.sendWithoutRetry(message);
            }
            finally
            {
                client.lock.unlock();
            }
        }
        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLen,
                            int type, CommandId cmdId) throws DisconnectedException
        {
            try
            {
                client.lock.lock();
                client.message.reset();
                client.message.setCommand(operation);
                if (cmdId != null) client.message.setCommandId(cmdId);
                client.message.getTopicRaw().set(topic, (int)topicOffset, (int)topicLen);
                if (type == Store.SOWDeleteByData)
                {
                    client.message.getDataRaw().set(data, (int)dataOffset, (int)dataLen);
                }
                else if (type == Store.SOWDeleteByFilter)
                {
                    client.message.getFilterRaw().set(data, (int)dataOffset, (int)dataLen);
                }
                else // SOWDeleteByKeys if (type == Store.SOWDeleteByKeys)
                {
                    client.message.getSowKeysRaw().set(data, (int)dataOffset, (int)dataLen);
                }
                client.message.setAckType(Message.AckType.Persisted|Message.AckType.Stats);
                client.message.setSequence(index);
                client.haSequenceNumber = index >= client.haSequenceNumber ? index+1 : client.haSequenceNumber;
                client.transport.sendWithoutRetry(message);
            }
            finally
            {
                client.lock.unlock();
            }
        }
    }

    class FailedWriteStoreReplayer implements StoreReplayer
    {
        int _reason;
        int _replayCount;
        public FailedWriteStoreReplayer(int reason)
        {
            _reason = reason;
            _replayCount = 0;
        }
        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLength,
                            byte[] corId, long corIdOff, long corIdLen)
        {
            ++_replayCount;
            _failedWriteHandler.failedWrite(index, operation,
                    topic, topicOffset, topicLen,
                    data, dataOffset, dataLength,
                    corId, corIdOff, corIdLen, _reason);
        }

        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLength,
                            byte[] corId, long corIdOff, long corIdLen, int expiration)
        {
            ++_replayCount;
            _failedWriteHandler.failedWrite(index, operation,
                    topic, topicOffset, topicLen,
                    data, dataOffset, dataLength,
                    corId, corIdOff, corIdLen, _reason);
        }

        public void execute(long index, int operation, byte[] topic, long topicOffset,
                            long topicLen, byte[] data, long dataOffset, long dataLength,
                            int expiration, CommandId cmdId)
        {
            ++_replayCount;
            _failedWriteHandler.failedWrite(index, operation,
                    topic, topicOffset, topicLen,
                    data, dataOffset, dataLength, null, 0, 0, _reason);
        }
        public int replayCount() { return _replayCount; }

    }

    private void cancelSynchronousWaiters(int connectionVersion)
    {
        ArrayList<CommandId> removeList = new ArrayList<CommandId>();

        lock.lock();
        try
        {
            for(Entry<CommandId, AckResponse> e : _acks.entrySet())
            {
                // collect items to remove and mark them abandoned.
                if(e.getValue().connectionVersion <= connectionVersion)
                {
                    e.getValue().abandoned = true;
                    removeList.add(e.getKey());
                }
            }
            // remove abandoned items from the list
            for(CommandId commandId : removeList)
            {
                _acks.remove(commandId);
            }
            ackReceived.signalAll();
        }
        finally
        {
            lock.unlock();
        }

    }

    private class StopWatch
    {
        private long _timeout;
        private long _start;

        public StopWatch()
        {
            _timeout = 0;
            _start = 0;
        }
        public boolean check()
        {
            if (_timeout == 0) return false;
            long current = System.currentTimeMillis();
            return ((current - _start) > _timeout);
        }
        public void start()
        {
            _start = System.currentTimeMillis();
        }
        public void setTimeout(long timeout)
        {
            _start = System.currentTimeMillis();
            _timeout = timeout;
        }
    }

}
