///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.io.RandomAccessFile;
import java.io.IOException;
import java.io.EOFException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.lang.Long;
import java.util.*;
import com.crankuptheamps.client.fields.Field;
import com.crankuptheamps.client.fields.BookmarkField;
import com.crankuptheamps.client.exception.*;

/** LoggedBookmarkStore implements a sequentially written log
 * of incoming and discarded messages.
 */
public class LoggedBookmarkStore implements BookmarkStore
{
    // Each entry begins with a single byte indicating the type of entry:
    // a new bookmark, or a discard of a previous one.
    static final byte ENTRY_BOOKMARK  = (byte)'b';
    static final byte ENTRY_DISCARD   = (byte)'d';
    static final byte ENTRY_PERSISTED = (byte)'p';

    // We use an in-memory array to remember the order in which bookmarks
    // arrive on a given subscription.  When the user has discard()ed all of the
    // bookmarks up to a certain bookmark B, we log B, and return it the next time
    // someone calls getMostRecent().
    static class Subscription implements com.crankuptheamps.client.Subscription
    {
        // This subscription's ID.
        Field _sub;

        // The last persisted bookmark if server version is > MIN_MULTI_BOOKMARK_VERSION
        private BookmarkField _lastPersisted;

        // A set of all of the undiscarded entries recovered that
        // had been received after the one returned by getMostRecent().
        // When these come in again, we won't log them a 2nd time.
        HashMap<Field, Long> _recovered = new HashMap<Field, Long>();

        // A map of active bookmark indices to their new location after a reconnect
        HashMap<Long, Long> _relocatedActiveBookmarks = new HashMap<Long, Long>();

        BookmarkRingBuffer _ring = new BookmarkRingBuffer();

        // The per-subscription memory of what we've seen from publishers
        HashMap<Long, Long> _publishers = new HashMap<Long, Long>();

        // The store that we log to when the time comes.
        LoggedBookmarkStore _parent;
        public Subscription()
        {
        }

        public void init(Field subscriptionId, LoggedBookmarkStore parent) throws IOException
        {
            _sub = subscriptionId.copy();
            _ring.setSubId(_sub);
            _parent = parent;
        }

        public synchronized long log(BookmarkField bookmark) throws IOException
        {
            // Check to see if this is a recovered bookmark.
            // If so, re-log in the new order.
            final Long recoveredValue = _recovered.remove(bookmark);
            if (recoveredValue != null)
            {
                long index = _ring.relog(recoveredValue, bookmark);
                _relocatedActiveBookmarks.put(recoveredValue, index);
                return index;
            }

            // Add this entry onto our list to remember the order it was seen.
            return _ring.log(bookmark);
        }

        public synchronized void discard(long index) throws IOException
        {
            long idx = index;
            Long newIndex = _relocatedActiveBookmarks.remove(index);
            // In case something's been recovered more than once, get full indirection
            while (newIndex != null)
            {
                _ring.discard(idx);
                idx = newIndex;
                newIndex = _relocatedActiveBookmarks.remove(newIndex);
            }
            BookmarkRingBuffer.Entry entry = _ring.getByIndex(idx);

            if(entry == null)
            {
                return;
            }

            _parent.write(_sub, LoggedBookmarkStore.ENTRY_DISCARD,
                  entry.getBookmark());
            _ring.discard(idx);
        }

        // Check to see if this message is older than the most recent one seen,
        // and if it is, check if it discarded.
        public synchronized boolean isDiscarded(BookmarkField bookmark) throws IOException
        {
            Long recoveredIndex = _recovered.get(bookmark);
            if (recoveredIndex != null)
            {
                BookmarkRingBuffer.Entry entry = _ring.getByIndex(recoveredIndex);
                // Need current active state
                boolean active = (entry == null) ? false : entry.isActive();
                // It's there, not yet discarded
                if (active)
                {
                    return false;
                }
                // It's either not there because it was discarded and persisted
                // and we cleaned it up on a previous incoming bookmark or it's
                // still there and discarded and persisted AND we don't have
                // anything new since recovery started.
                if ((entry == null || (entry.isPersisted() && !active)) &&
                    _ring.getStartIndex() == BookmarkRingBuffer.UNSET_INDEX)
                {
                    _ring.discard(recoveredIndex); // Updates most recent
                    _parent.write(_sub, LoggedBookmarkStore.ENTRY_DISCARD, bookmark);
                }
                else // It's either not persisted or already in flight so relog
                {
                    long newIndex = _ring.relog(recoveredIndex, bookmark);
                    _parent.write(_sub, LoggedBookmarkStore.ENTRY_BOOKMARK, bookmark);
                    // Check if we're relocating because it's still out for processing
                    if (active)
                    {
                        _relocatedActiveBookmarks.put(recoveredIndex, newIndex);
                    }
                }
                _recovered.remove(bookmark);
                return true;
            }
            long publisher = bookmark.getPublisherId();
            long sequence = bookmark.getSequenceNumber();
            if(!_publishers.containsKey(publisher) || _publishers.get(publisher) < sequence)
            {
                _publishers.put(publisher, sequence);
                return false;
            }
            // During failure and recovery scenarios, we'll see out of order
            // bookmarks arrive, either (a) because we're replaying or (b)
            // because a publisher has cut over, and we've cut over to a new server.
            // Scan the list to see if we have a match.
            BookmarkRingBuffer.Entry entry = _ring.find(bookmark);

            if(entry != null)
            {
                return !entry.isActive();
            }

            return true; // message is totally discarded
        }

        public synchronized Field getMostRecent()
        {
            // _recent is the most recent bookmark.
            // when this is called, we'll take a moment to update the list of things recovered,
            // so we don't accidentally log anything we ought not to.
            updateRecovery();
            return _ring.getLastDiscarded();
        }

        public synchronized Field getMostRecentList()
        {
            // when this is called, we'll take a moment to update the list of things
            // recovered, so we don't accidentally log anything we ought not to.
            updateRecovery();
            BookmarkField lastDiscarded = (BookmarkField)_ring.getLastDiscarded();
            boolean useLastDiscarded = (lastDiscarded != null && !lastDiscarded.isNull());
            long lastDiscardedPub = 0;
            long lastDiscardedSeq = 0;
            boolean useLastPersisted = (_lastPersisted != null && !_lastPersisted.isNull());
            long lastPersistedPub = 0;
            long lastPersistedSeq = 0;
            if (useLastPersisted)
            {
                lastPersistedPub = ((BookmarkField)_lastPersisted).getPublisherId();
                lastPersistedSeq = ((BookmarkField)_lastPersisted).getSequenceNumber();
            }
            if (useLastDiscarded)
            {
                if (_ring.isEmpty() && useLastPersisted)
                {
                    useLastDiscarded = false;
                }
                else
                {
                    lastDiscardedPub = lastDiscarded.getPublisherId();
                    lastDiscardedSeq = lastDiscarded.getSequenceNumber();
                    // Only use one if they are same publisher
                    if (useLastPersisted && (lastDiscardedPub == lastPersistedPub))
                    {
                        useLastPersisted = (lastPersistedSeq < lastDiscardedSeq);
                        useLastDiscarded = !useLastPersisted;
                    }
                }
            }
            StringBuilder recentStr = new StringBuilder();
            if (useLastDiscarded)
            {
                recentStr.append((lastDiscarded).getValue(Charset.forName("ISO-8859-1").newDecoder()));
            }
            if (!useLastPersisted && !useLastDiscarded)
            {
                if (_publishers.isEmpty())
                {
                    // Set last persisted to EPOCH and return it
                    _lastPersisted.setValue(Client.Bookmarks.EPOCH, Charset.forName("ISO-8859-1").newEncoder());
                    return _lastPersisted;
                }
                Iterator it = _publishers.entrySet().iterator();
                while (it.hasNext())
                {
                    if (recentStr.length() > 0) recentStr.append(",");
                    Map.Entry pairs = (Map.Entry)it.next();
                    long pubId = (Long)pairs.getKey();
                    long seq = (Long)pairs.getValue();
                    recentStr.append(pubId).append("|");
                    recentStr.append(seq).append("|");
                }
            }
            if (useLastPersisted)
            {
                if (recentStr.length() > 0) recentStr.append(",");
                recentStr.append(((BookmarkField)_lastPersisted).getValue(Charset.forName("ISO-8859-1").newDecoder()));
            }
            BookmarkField recentList = new BookmarkField();
            recentList.setValue(recentStr.toString(), Charset.forName("ISO-8859-1").newEncoder());
            return recentList;
        }

        private void updateRecovery()
        {
            _recovered.clear();
            long end = _ring.getEndIndex();
            for(long index = _ring.setRecovery(); index< end; ++index)
            {
                BookmarkRingBuffer.Entry entry = _ring.getByIndex(index);
                if(entry != null && entry._bookmark != null && !entry._bookmark.isNull())
                {
                    _recovered.put(entry.getBookmark(), index);
                }
            }
        }

        public synchronized void persisted(long bookmark) throws IOException
        {
            BookmarkRingBuffer.Entry entry = _ring.getByIndex(bookmark);
            if (entry == null)
                return;
            BookmarkField bookmarkField = entry.getBookmark();
            _parent.write(_sub, ENTRY_PERSISTED, bookmarkField);
            _ring.persisted(bookmark);
        }

        public synchronized void persisted(BookmarkField bookmark) throws IOException
        {
            if (bookmark == null || bookmark.isNull()) return;
            _parent.write(_sub, ENTRY_PERSISTED, bookmark);
            _ring.persisted(bookmark);
        }

        public synchronized void setLastPersisted(long bookmark) throws IOException
        {
            BookmarkRingBuffer.Entry entry = _ring.getByIndex(bookmark);
            if (entry == null)
                return;
            BookmarkField bookmarkField = entry.getBookmark();
            _parent.write(_sub, ENTRY_PERSISTED, bookmarkField);
            if (_lastPersisted != null) _lastPersisted.reset();
            _lastPersisted = bookmarkField.copy();
        }

        public synchronized void setLastPersisted(BookmarkField bookmark) throws IOException
        {
            if (bookmark == null || bookmark.isNull()) return;
            if (!_ring.persistedAcks() && _lastPersisted != null &&
                bookmark.getPublisherId() == _lastPersisted.getPublisherId() &&
                bookmark.getSequenceNumber() <= _lastPersisted.getSequenceNumber())
            {
                return;
            }
            if (_lastPersisted != null) _lastPersisted.reset();
            _parent.write(_sub, ENTRY_PERSISTED, bookmark);
            _lastPersisted = bookmark.copy();
        }

        public synchronized void noPersistedAcks()
        {
            _ring.noPersistedAcks();
        }

        public synchronized void setPersistedAcks()
        {
            _ring.setPersistedAcks();
        }

        public synchronized long getOldestBookmarkSeq()
        {
            return _ring.getStartIndex();
        }

        public void setResizeHandler(BookmarkStoreResizeHandler handler, BookmarkStore store)
        {
            _ring.setResizeHandler(handler, store);
        }
    }

    HashMap<Field, Subscription> _subs = new HashMap<Field, Subscription>();
    HashSet<Field> _noPersistedAcks = new HashSet<Field>();
    RandomAccessFile _file;
    boolean _recovering = false;
    Pool<Subscription> _pool;
    BookmarkStoreResizeHandler _resizeHandler = null;
    private int _serverVersion = Client.MIN_MULTI_BOOKMARK_VERSION;

    public LoggedBookmarkStore(String path) throws IOException
    {
        this(path, 1);
    }
    public LoggedBookmarkStore(String path, int targetNumberOfSubscriptions) throws IOException
    {
        _pool = new Pool<Subscription>(Subscription.class, targetNumberOfSubscriptions);
        _file = new RandomAccessFile(path, "rw");

        try
        {
            recover();
        }
        catch(IOException ioex)
        {
            try
            {
                _file.close();
            }
            catch (IOException ignoreTheCloseException)
            {
            }
            finally
            {
                _file = null;
            }
            throw ioex;
        }
    }
    void write(Field sub, byte entry, Field data) throws IOException
    {
        if(!_recovering)
        {
            synchronized(this)
            {
                _file.writeByte(sub.length);
                _file.write(sub.buffer, sub.position, sub.length);
                _file.writeByte(entry);
                _file.writeByte(data.length);
                _file.write(data.buffer, data.position, data.length);
                _file.writeByte((byte)'\n');
            }
        }
    }
    void write(Field sub, byte entry, long data) throws IOException
    {
        if(!_recovering)
        {
            synchronized(this)
            {
                _file.writeByte(sub.length);
                _file.write(sub.buffer, sub.position, sub.length);
                _file.writeByte(entry);
                _file.writeLong(data);
                _file.writeByte((byte)'\n');
            }
        }
    }

    private void recover() throws IOException
    {
        _recovering = true;
        HashMap<Field, HashMap<BookmarkField,Long>> ids = new HashMap<Field, HashMap<BookmarkField,Long>>();
        HashMap<Field, ArrayList<BookmarkField>> removeMap = new HashMap<Field, ArrayList<BookmarkField>>();
        ArrayList<BookmarkField> removeRemoveList = new ArrayList<BookmarkField>();
        byte[] sub = new byte[255];
        byte[] bookmark = new byte[255];
        long lastGoodPosition = _file.getFilePointer();
        try
        {
            while(lastGoodPosition < _file.length())
            {
                Field subId = new Field();
                BookmarkField bookmarkField = new BookmarkField();
                long position = _file.getFilePointer();
                byte subLen = _file.readByte();
                for(int i = 0; i<subLen; i++)
                {
                    sub[i] = _file.readByte();
                }
                subId.set(sub, 0, subLen);
                HashMap<BookmarkField, Long> subscriptionMap = ids.get(subId);
                if(subscriptionMap == null)
                {
                    subscriptionMap = new HashMap<BookmarkField, Long>();
                    ids.put(subId.copy(), subscriptionMap);
                }
                ArrayList<BookmarkField> removeList = removeMap.get(subId);
                if(removeList == null)
                {
                    removeList = new ArrayList<BookmarkField>();
                    removeMap.put(subId.copy(), removeList);
                }
                Subscription subscription = find(subId);
                
                switch(_file.readByte())
                {
                case -1:
                    // done
                    return;
                case ENTRY_BOOKMARK:
                    int len = _file.readByte();
                    if(len == -1 || len > (_file.length() - _file.getFilePointer()))
                    {
                        // truncated entry
                        _file.seek(position);
                        return;
                    }
                    _file.read(bookmark, 0, len);
                    bookmarkField.set(bookmark, 0, len);
                    if(subscriptionMap.get(bookmarkField) != null)
                    {
                        subscription.getMostRecent();
                        subscriptionMap.clear();
                    }
                    if(!subscription.isDiscarded(bookmarkField))
                    {
                        long addedIndex = subscription.log(bookmarkField);
                        subscriptionMap.put((BookmarkField)bookmarkField.copy(), addedIndex);
                    }
                    break;

                case ENTRY_DISCARD:
                    len = _file.readByte();
                    if(len == -1 || len > (_file.length() - _file.getFilePointer()))
                    {
                        // truncated entry
                        _file.seek(position);
                        return;
                    }
                    _file.read(bookmark, 0, len);
                    bookmarkField.set(bookmark, 0, len);

                    Long subscriptionMapEntry = subscriptionMap.get(bookmarkField);
                    if(subscriptionMapEntry != null)
                    {
                        // don't remove from the map until we've PERSISTED
                        removeList.add(bookmarkField.copy());
                        subscription.discard(subscriptionMapEntry);
                    }
                    break;

                case ENTRY_PERSISTED:
                    len = _file.readByte();
                    if(len == -1 || len > (_file.length() - _file.getFilePointer()))
                    {
                        // truncated entry
                        _file.seek(position);
                        return;
                    }
                    _file.read(bookmark, 0, len);
                    bookmarkField.set(bookmark, 0, len);

                    if(subscriptionMap.containsKey(bookmarkField))
                    {
                        long discardIndex = subscriptionMap.get(bookmarkField);
                        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
                            subscription.setLastPersisted(discardIndex);
                        else
                            subscription.persisted(discardIndex);
                        // empty out the subscriptionMap for everything in the removeList.
                        removeRemoveList.clear();
                        for(BookmarkField bf:removeList)
                        {
                            Long subMapEntry = subscriptionMap.get(bf);
                            if(subMapEntry!=null && subMapEntry <= discardIndex)
                            {
                                removeRemoveList.add(bf);
                            }
                        }
                        for(BookmarkField bf:removeRemoveList)
                        {
                            removeList.remove(bf);
                            subscriptionMap.remove(bf);
                        }
                    }
                    break;

                default:
                    throw new IOException("Corrupt file found.");
                }
                _file.readByte();
                lastGoodPosition = _file.getFilePointer();
            }
        }
        catch (IOException ex)
        {
            if (lastGoodPosition > 0)
            {
                _file.seek(lastGoodPosition);
            }
            else
            {
                throw ex;
            }
        }
        finally
        {
            _recovering = false;
        }
    }

    public long log(Message message) throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {

            BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
            Subscription sub = (LoggedBookmarkStore.Subscription)message.getSubscription();
            Field subId = message.getSubIdRaw();
            if (subId == null || subId.isNull())
                subId = message.getSubIdsRaw();

            // log the arrival of this bookmark.
            write(subId, LoggedBookmarkStore.ENTRY_BOOKMARK, bookmark);

            if (sub == null)
            {
                sub = find(subId);
                message.setSubscription(sub);
            }
            long index = sub.log(bookmark);
            message.setBookmarkSeqNo(index);
            if (_noPersistedAcks.contains(subId))
                persisted(subId, bookmark);
            return index;
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error logging to bookmark store", ioex);
        }
    }

    public void discard(Field subId, long bookmarkSeqNo) throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {
            find(subId).discard(bookmarkSeqNo);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error discarding from bookmark store", ioex);
        }
    }

    public void discard(Message message) throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {
            long bookmark = message.getBookmarkSeqNo();
            Subscription sub = (LoggedBookmarkStore.Subscription)message.getSubscription();
            if (sub == null)
            {
                Field subId = message.getSubIdRaw();
                if (subId == null || subId.isNull())
                    subId = message.getSubIdsRaw();
                sub = find(subId);
                message.setSubscription(sub);
            }
            sub.discard(bookmark);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error discarding from bookmark store", ioex);
        }
    }

    public Field getMostRecent(Field subId) throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {
            if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
                return find(subId).getMostRecentList();
            else
                return find(subId).getMostRecent();
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error getting most recent from bookmark store", ioex);
        }
    }

    public boolean isDiscarded(Message message) throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {
            BookmarkField bookmark = (BookmarkField)message.getBookmarkRaw();
            Subscription sub = (LoggedBookmarkStore.Subscription)message.getSubscription();
            if (sub == null)
            {
                Field subId = message.getSubIdRaw();
                if (subId == null || subId.isNull())
                    subId = message.getSubIdsRaw();
                sub = find(subId);
                message.setSubscription(sub);
            }
            return sub.isDiscarded(bookmark);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error checking is discarded in bookmark store", ioex);
        }
    }

    public void persisted(Field subId, long bookmark) throws AMPSException
    {
        try
        {
            if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
                find(subId).setLastPersisted(bookmark);
            else
                find(subId).persisted(bookmark);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error logging persisted to bookmark store", ioex);
        }
    }

    public void persisted(Field subId, BookmarkField bookmark) throws AMPSException
    {
        try
        {
            if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION)
                find(subId).setLastPersisted(bookmark);
            else
                find(subId).persisted(bookmark);
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error logging persisted to bookmark store", ioex);
        }
    }

    public void noPersistedAcks(Field subId) throws AMPSException
    {
        try
        {
            Subscription sub = find(subId);
            _noPersistedAcks.add(subId.copy());
            sub.noPersistedAcks();
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error setting no persisted acks on bookmark store", ioex);
        }
    }

    public synchronized long getOldestBookmarkSeq(Field subId) throws AMPSException
    {
        long retVal = 0;
        try
        {
            retVal = find(subId).getOldestBookmarkSeq();
        }
        catch (IOException ioex)
        {
            throw new AMPSException("Error getting oldest bookmark seq from bookmark store", ioex);
        }
        return retVal;
    }

    public void setResizeHandler(BookmarkStoreResizeHandler handler)
    {
        _resizeHandler = handler;
        Iterator it = _subs.entrySet().iterator();
        while (it.hasNext())
        {
            Map.Entry pairs = (Map.Entry)it.next();
            ((Subscription)pairs.getValue()).setResizeHandler(handler, this);
        }
    }

    private synchronized Subscription find(Field subId) throws IOException
    {
        Subscription s = _subs.get(subId);
        if(s==null)
        {
            s=_pool.get();
            s.init(subId, this);
            s.setResizeHandler(_resizeHandler, this);
            if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION ||
                _serverVersion < Client.MIN_PERSISTED_BOOKMARK_VERSION)
            {
                s.noPersistedAcks();
            }
            _subs.put(subId.copy(), s);
        }
        return s;
    }


    /*
    public void compactTo(String path) throws IOException
    {
        // Create the file
        // for each Subscription s in self:
            // write a ENTRY_BOOKMARK for s.getMostRecent()
            // write a ENTRY_DISCARD for s.getMostRecent()

            // for each entry e in s:
                // write a ENTRY_BOOKMARK for e
                // if !e._active: write a ENTRY_DISCARD for e
    }
    */

    public synchronized void purge() throws AMPSException
    {
        // delete the file on disk.
        try
        {
            _file.setLength(0);
        }
        catch(IOException ioex)
        {
            throw new StoreException("Error truncating file", ioex);
        }

        _subs = new HashMap<Field, Subscription>();
        _noPersistedAcks = new HashSet<Field>();
    }

    public synchronized void close() throws AMPSException
    {
        if(_file == null)
        {
            throw new StoreException("Store not open.");
        }
        try
        {
            _file.close();
        }
        catch(IOException ioex)
        {
            throw new StoreException("Error closing file", ioex);
        }
        finally
        {
            _file = null;
        }

    }

    public void setServerVersion(int version)
    {
        if (_serverVersion == version) return;
        _serverVersion = version;
        if (_serverVersion >= Client.MIN_MULTI_BOOKMARK_VERSION ||
            _serverVersion < Client.MIN_PERSISTED_BOOKMARK_VERSION)
        {
            Iterator it = _subs.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry pairs = (Map.Entry)it.next();
                ((Subscription)pairs.getValue()).noPersistedAcks();
            }
        }
        else
        {
            Iterator it = _subs.entrySet().iterator();
            while (it.hasNext())
            {
                Map.Entry pairs = (Map.Entry)it.next();
                ((Subscription)pairs.getValue()).setPersistedAcks();
            }
        }
    }
}
