////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.nio.ByteBuffer;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;

import com.crankuptheamps.client.fields.*;
import com.crankuptheamps.client.exception.CommandException;

public abstract class Message
{
    // Not using enums here looks like Java 1.4 circa 2003.
    //   That said, it's fast, low-cost, and safe enough if
    //   guarded against mis-use (which we do.)
    static public final class Command
    {
        public static final int Unknown                 =      0;
        public static final int Publish                 =      1;
        public static final int Subscribe               =      2;
        public static final int Unsubscribe             =      4;
        public static final int SOW                     =      8;
        public static final int Heartbeat               =     16;
        public static final int SOWDelete               =     32;
        public static final int DeltaPublish            =     64;
        public static final int Logon                   =    128;
        public static final int SOWAndSubscribe         =    256;
        public static final int DeltaSubscribe          =    512;
        public static final int SOWAndDeltaSubscribe    =   1024;
        public static final int StartTimer              =   2048;
        public static final int StopTimer               =   4096;
        public static final int GroupBegin              =   8192;
        public static final int GroupEnd                =  16384;
        public static final int OOF                     =  32768;
        public static final int Ack                     =  65536;
        public static final int Flush                   = 131072;
    }

    // These AckTypes are often combined for a single command
    //   so we favor the int 'mask' versus a heavy EnumSet.
    static public final class AckType
    {
        public static final int None       =  0;
        public static final int Received   =  1;
        public static final int Parsed     =  2;
        public static final int Persisted  =  4;
        public static final int Processed  =  8;
        public static final int Completed  = 16;
        public static final int Stats      = 32;
    }

    static public final class Status
    {
        public static final int None      = 0;
        public static final int Success   = 1;
        public static final int Failure   = 2;
        public static final int Retry     = 3;
    }

    static public final class Reason
    {
        public static final int None                        = 0;
        public static final int Duplicate                   = 1;
        public static final int BadFilter                   = 2;
        public static final int BadRegexTopic               = 3;
        public static final int SubscriptionAlreadyExists   = 4;
        public static final int Deleted                     = 5;
        public static final int Expired                     = 6;
        public static final int Match                       = 7;
        public static final int InvalidTopic                = 8;
        public static final int NameInUse                   = 9;
        public static final int AuthFailure                 = 10;
        public static final int NotEntitled                 = 11;
        public static final int AuthDisabled                = 12;
        public static final int InvalidBookmark             = 13;
        public static final int InvalidOrderBy              = 14;
        public static final int SubidInUse                  = 15;
        public static final int Other                       = 16;
    }

    /**
     * Represents the options for an AMPS command.
     * Options are provided as a comma-delimited list. For example,
     * to provide the options OOF and NoEmpties to a SOW and Subscribe
     * command, you can use the following code:
     * <pre>
     * {@code
     * Command c = new Command("sow_and_subscribe")
     *                   .setOptions(Message.Options.OOF +
                                     Message.Options.NoEmpties)
                         // other parameters for command here
                         ; 
     *}
     * </pre>
     */
    static public final class Options
    {
        public static final String None             = null;
        public static final String Live             = "live,";
        public static final String OOF              = "oof,";
        public static final String Replace          = "replace,";
        public static final String NoEmpties        = "no_empties,";
        public static final String SendKeys         = "send_keys,";
    }

    protected final CharsetEncoder encoder;
    protected final CharsetDecoder decoder;

    static protected final int MINIMUM_SERVER_VERSION = 4000000;

    public Message(CharsetEncoder encoder,
                   CharsetDecoder decoder)
    {
        this.encoder = encoder;
        this.decoder = decoder;
    }


    long _bookmarkSeqNo = 0;

    /**
     * Sets the bookmark sequence number for this message. The bookmark
     * sequence number is used by bookmark stores to track messages.
     */
    public void setBookmarkSeqNo(long val)
    {
        _bookmarkSeqNo = val;
    }
    /**
     * Returns the bookmark sequence number for this message. The bookmark
     * sequence number is used by bookmark stores to track messages.
     */

    public long getBookmarkSeqNo()
    {
        return _bookmarkSeqNo;
    }

    Subscription _subscription = null;

    /**
     * Sets the subscription for this message. The subscription is used
     * by bookmark stores to track messages.
     */
    public void setSubscription(Subscription subscription)
    {
        _subscription = subscription;
    }
    /**
     * Gets the subscription for this message. The subscription is used
     * by bookmark stores to track messages.
     */
    public Subscription getSubscription()
    {
        return _subscription;
    }

    private byte[]   buffer = null;
    /**
     * Sets the byte buffer used by the message. It's not common for
     * applications to use this directly.
     *
     *  @param  buffer the byte buffer to use for the message.
     *
     */
    public void setBuffer(byte[] buffer)
    {
        this.buffer = buffer;
    }
    /**
     * Gets the byte buffer used by the message.
     *
     *  @return the raw byte buffer backing this message object.
     *
     */
    public byte[] getBuffer()
    {
        return this.buffer;
    }

    private int _rawBufferOffset = 0;
    /**
     * Sets the offset into the byte buffer where this message begins.
     *
     *  @param offset offset of first byte of message within the buffer.
     *
     */
    public void setRawBufferOffset(int offset)
    {
        this._rawBufferOffset = offset;
    }
    /**
     * Gets the offset into the byte buffer where this message begins.
     *
     *  @return offset of first byte of message within the buffer.
     *
     */
    public int  getRawBufferOffset()
    {
        return this._rawBufferOffset;
    }

    private int _rawBufferLength = 0;
    /**
     * Sets the length of the message within the byte buffer.
     *
     *  @param length length of message within the buffer.
     *
     */
    public void setRawBufferLength(int length)
    {
        this._rawBufferLength = length;
    }
    /**
     * Gets the length of the message within the byte buffer.
     *
     *  @return length of message within the buffer.
     *
     */
    public int  getRawBufferLength()
    {
        return this._rawBufferLength;
    }

    static enum SerializationResult { OK, BufferTooSmall }
    abstract SerializationResult serialize(ByteBuffer buffer);
    
    public void reset()
    {
        _AckType.reset();
        _BatchSize.reset();
        _Bookmark.reset();
        _ClientName.reset();
        _CommandId.reset();
        _Command.reset();
        _CorrelationId.reset();
        _Data.reset();
        _Expiration.reset();
        _Filter.reset();
        _OrderBy.reset();
        _GroupSeqNo.reset();
        _Length.reset();
        _Matches.reset();
        _MaxMessages.reset();
        _MessageId.reset();
        _Options.reset();
        _QueryId.reset();
        _Reason.reset();
        _RecordsInserted.reset();
        _RecordsUpdated.reset();
        _RecordsDeleted.reset();
        _RecordsReturned.reset();
        _SendEmpties.reset();
        _SendMatchingIds.reset();
        _SendOOF.reset();
        _Sequence.reset();
        _SowKey.reset();
        _SowKeys.reset();
        _Status.reset();
        _SubId.reset();
        _SubIds.reset();
        _Timestamp.reset();
        _TopN.reset();
        _Topic.reset();
        _TopicMatches.reset();
        _UserId.reset();
        _Version.reset();
        _Password.reset();
        _bookmarkSeqNo = 0;
        _subscription = null;
    }


    AckTypeField _AckType = new AckTypeField();
    /**
     *  Returns the type of acknowledgement for an acknowledgement message.
     */
    public int getAckType()
    {
        return _AckType.getValue();
    }
    /**
     *  Sets the type of acknowledgement for an acknowledgement message.
     */
    public void setAckType(int v)
    {
        _AckType.setValue(v);
    }

    /**
     * Sets the type of acknowledgement for an acknowlegdement message.
     */
    public void setAckType(String v)
    {
        _AckType.set(v.getBytes());
    }
    IntegerField _BatchSize = new IntegerField();
    /**
     * Returns the batch size for this message.
     */
    public int getBatchSize()
    {
        return _BatchSize.getValue();
    }
    /**
     * Sets the batch size for this message.
     */
    public void setBatchSize(int v)
    {
        _BatchSize.setValue(v);
    }

    BookmarkField _Bookmark = new BookmarkField();
    /**
     * Returns the AMPS bookmark for this message. The bookmark is an
     * idenfitier assigned by AMPS to locate a message in the transaction
     * log.
     */
    public String getBookmark()
    {
        return _Bookmark.getValue(decoder);
    }

    /**
     * Gets the raw value for the AMPS bookmark. The bookmark is an identifier
     * assigned by AMPS to locate a message in the transaction log.
     */
    public BookmarkField getBookmarkRaw()
    {
        return _Bookmark;
    }
    /**
     * Sets the value for the AMPS bookmark. The bookmark is an identifier
     * assigned by AMPS to locate a message in the transaction log. Setting
     * the bookmark is used for commands that locate messages in the log,
     * such as a bookmark subscribe or historical SOW query. Setting the
     * bookmark on a publish message has no effect.
     */
    public void setBookmark(String v)
    {
        _Bookmark.setValue(v, encoder);
    }
    /**
     * Sets the value for the AMPS bookmark. The bookmark is an identifier
     * assigned by AMPS to locate a message in the transaction log. Setting
     * the bookmark is used for commands that locate messages in the log,
     * such as a bookmark subscribe or historical SOW query. Setting the
     * bookmark on a publish message has no effect.
     */
    public void setBookmark(byte[] buffer,int offset,int length)
    {
        _Bookmark.setValue(buffer,offset,length);
    }

    StringField _ClientName = new StringField();
    /**
     * Gets the client name on this message.
     */
    public String getClientName()
    {
        return _ClientName.getValue(decoder);
    }
    /**
     * Gets the client name on this message.
     */
    public Field getClientNameRaw()
    {
        return _ClientName;
    }
    /**
     * Sets the name of the client sending the message.
     */
    public void setClientName(String v)
    {
        _ClientName.setValue(v, encoder);
    }
    /**
     * Sets the name of the client sending the message.
     */
    public void setClientName(byte[] buffer,int offset,int length)
    {
        _ClientName.setValue(buffer,offset,length);
    }

    StringField _CommandId = new StringField();
    /**
     * Gets the CommandId on this message. The CommandId is an identifier
     * set by the client that is used to correlate later messages. For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe.
     */
    public String getCommandId()
    {
        return _CommandId.getValue(decoder);
    }
    /**
     * Gets the CommandId on this message. The CommandId is an identifier
     * set by the client that is used to correlate later messages. For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe.
     */
    public Field getCommandIdRaw()
    {
        return _CommandId;
    }
    /**
     * Gets the CommandId on this message by copying it into the provided
     * CommandId object. The CommandId is an identifier
     * set by the client that is used to correlate later messages.
     * For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe. The CommandId is returned on
     * ack messages in response to the command.
     * @param v the object to hold the value of the CommandId
     */
    public boolean getCommandId(CommandId v)
    {
        return _CommandId.getValue(v);
    }
    /**
     * Sets the CommandId on this message. The CommandId is an identifier
     * set by the client that is used to correlate later messages and commands.
     * For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe. The CommandId is returned on
     * ack messages in response to the command.
     */
    public void setCommandId(String v)
    {
        _CommandId.setValue(v,encoder);
    }

    /**
     * Sets the CommandId on this message. The CommandId is an identifier
     * set by the client that is used to correlate later messages and commands.
     * For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe. The CommandId is returned on
     * ack messages in response to the command.
     */
    public void setCommandId(CommandId v)
    {
        _CommandId.setValue(v);
    }

    /**
     * Sets the CommandId on this message. The CommandId is an identifier
     * set by the client that is used to correlate later messages and commands.
     * For example,
     * the client sets a CommandId on a subscription request to AMPS, and can
     * later use that CommandId to unsubscribe. The CommandId is returned on
     * ack messages in response to the command.
     */
    public void setCommandId(byte[] buffer,int offset,int length)
    {
        _CommandId.setValue(buffer,offset,length);
    }

    CommandField _Command= new CommandField();

    /**
     * Returns the command for this message, indicating the type of
     * of message this is. The fields that are supported by this message,
     * and how they are set and interpreted, depend on the type of message.
     * See the AMPS Command Reference for details.
     */
    public int getCommand()
    {
        return _Command.getValue();
    }
     /**
     * Sets the command for this message, indicating the type of
     * of message this is. The fields that are supported by this message,
     * and how they are set and interpreted, depend on the type of message.
     * See the AMPS Command Reference for details.
     * @param v the command to set on this message
     */
    public void setCommand(int v)
    {
        _Command.setValue(v);
    }

     /**
     * Sets the command for this message, indicating the type of
     * of message this is. The fields that are supported by this message,
     * and how they are set and interpreted, depend on the type of message.
     * See the AMPS Command Reference for details.
     * @param v the command to set on this message. The value provided must be a value supported by the AMPS server, or the server will refuse to process the command.
     */
    public void setCommand(String v)
    {
        _Command.set(v.getBytes());
    }

    StringField _Data = new StringField();
    /**
     * Returns the payload of the message.
     */
    public String getData()
    {
        return _Data.getValue(decoder);
    }
    /**
     * Returns the payload of the message.
     */
    public Field getDataRaw()
    {
        return _Data;
    }

    /**
     * Sets the payload of the message.
     */
    public void setData(String v)
    {
        _Data.setValue(v,encoder);
    }
    /**
     * Sets the payload of the message.
     */
    public void setData(byte[] buffer,int offset,int length)
    {
        _Data.setValue(buffer,offset,length);
    }

    IntegerField _Expiration = new IntegerField();
    /**
     *  Returns the expiration set for this message.
     */
    public int getExpiration()
    {
        return _Expiration.getValue();
    }
    /**
     *  Sets the expiration set for this message.
     *  The expiration is used on a publish command to set the lifetime
     *  of a message. For the lifetime to be processed by AMPS, the
     * message must be published to a SOW topic that supports message
     * expiration. See the AMPS User Guide for details.
     * @param v the lifetime of the message
     */
    public void setExpiration(int v)
    {
        _Expiration.setValue(v);
    }

    /**
     * Returns the filter on this message.
     * Filters are used for commands that provide content filtering,
     * such as SOW queries, subscriptions, and SOW delete.
     */
    StringField _Filter = new StringField();
    public String getFilter()
    {
        return _Filter.getValue(decoder);
    }
    /**
     * Returns the filter on this message.
     * Filters are used for commands that provide content filtering,
     * such as SOW queries, subscriptions, and SOW delete.
     */
   public Field getFilterRaw()
    {
        return _Filter;
    }
    /**
     * Sets the filter on this message.
     * Filters are used for commands that provide content filtering,
     * such as SOW queries, subscriptions, and SOW delete.
     */
    public void setFilter(String v)
    {
        _Filter.setValue(v,encoder);
    }
    /**
     * Sets the filter on this message.
     * Filters are used for commands that provide content filtering,
     * such as SOW queries, subscriptions, and SOW delete.
     */
    public void setFilter(byte[] buffer,int offset,int length)
    {
        _Filter.setValue(buffer,offset,length);
    }

    StringField _OrderBy = new StringField();
    /**
     * Get the OrderBy parameter for this message.
     * For commands that support ordering, this parameter provides the
     * order in which AMPS returns results.
     */
    public String getOrderBy()
    {
        return _OrderBy.getValue(decoder);
    }
    /**
     * Get the OrderBy parameter for this message.
     * For commands that support ordering, this parameter provides the
     * order in which AMPS returns results.
     */
    public Field getOrderByRaw()
    {
        return _OrderBy;
    }
    /**
     * Set the OrderBy parameter for this message.
     * For commands that support ordering, this parameter provides the
     * order in which AMPS returns results. See the AMPS User Guide for
     * details.
     * @param v the ordering expression
     */
    public void setOrderBy(String v)
    {
        _OrderBy.setValue(v,encoder);
    }
    /**
     * Set the OrderBy parameter for this message.
     * For commands that support ordering, this parameter provides the
     * order in which AMPS returns results. See the AMPS User Guide for
     * details.
     * @param buffer the buffer that contains the expression
     * @param offset the location at which the expression begins
     * @param length the length of the expression
     */
    public void setOrderBy(byte[] buffer,int offset,int length)
    {
        _OrderBy.setValue(buffer,offset,length);
    }


   
    LongField _GroupSeqNo = new LongField();
    /**
     * Get the group sequence number. The group sequence number is the number
     * of the batch within the SOW response.
     */
    public long getGroupSeqNo()
    {
        return _GroupSeqNo.getValue();
    }
    /**
     * Set the group sequence number. The group sequence number is the number
     * of the batch within the SOW response, and is set on incoming messages
     * by the AMPS client.
     */
    public void setGroupSeqNo(long v)
    {
        _GroupSeqNo.setValue(v);
    }

    LongField _Matches = new LongField();
   /**
    * Returns the number of matches in the command this message acknowledges.
    * This header is provided on an acknowledgement message that includes
    * information on the number of matches. See the AMPS Command Reference for
    * details.
    */
    public long getMatches()
    {
        return _Matches.getValue();
    }
    /**
     * Sets the matches field on this message. This header is typically
     * provided by AMPS.
     * @param v the value to set
     */
    public void setMatches(long v)
    {
        _Matches.setValue(v);
    }

    LongField _MaxMessages = new LongField();
    /**
     * Gets the max messages header, unused in recent versions of AMPS.
     * @deprecated This header is no longer used by AMPS.
     */
    public long getMaxMessages()
    {
        return _MaxMessages.getValue();
    }
    /**
     * Sets the max messages header, unused in recent versions of AMPS.
     *
     * @deprecated This header is no longer used by AMPS.
     */
    public void setMaxMessages(long v)
    {
        _MaxMessages.setValue(v);
    }

    StringField _MessageId = new StringField();
    /**
     * Gets the max messages header, unused in recent versions of AMPS.
     * @deprecated This header is no longer used by AMPS.
     */
    public String getMessageId()
    {
        return _MessageId.getValue(decoder);
    }
    /**
     * Gets the max messages header, unused in recent versions of AMPS.
     * @deprecated This header is no longer used by AMPS.
     */
    public Field getMessageIdRaw()
    {
        return _MessageId;
    }
    /**
     * Sets the max messages header, unused in recent versions of AMPS.
     *
     * @deprecated This header is no longer used by AMPS.
     */    public void setMessageId(String v)
    {
        _MessageId.setValue(v,encoder);
    }
    /**
     * Sets the max messages header, unused in recent versions of AMPS.
     *
     * @deprecated This header is no longer used by AMPS.
     */
    public void setMessageId(byte[] buffer,int offset,int position)
    {
        _MessageId.setValue(buffer,offset,position);
    }


    StringField _Password = new StringField();

    /**
     * Returns the password set on the message.
     * This field is typically used during the logon sequence.
     */
    public String getPassword()
    {
        return _Password.getValue(decoder);
    }
    /**
     * Returns the password set on the message.
     * This field is used during the logon sequence.
     */
    public Field getPasswordRaw()
    {
        return _Password;
    }
   /**
    * Set the password on the message.
    * This field is typically used during the logon sequence.
    * @param v the password to set
    */
    public void setPassword(String v)
    {
        _Password.setValue(v,encoder);
    }
   /**
    * Set the password on the message.
    * This field is typically used during the logon sequence.
    * @param buffer the buffer that contains the password
    * @param offset the location where the password starts
    * @param length the length of the password
    */
    public void setPassword(byte[] buffer,int offset,int length)
    {
        _Password.setValue(buffer,offset,length);
    }

    IntegerField _Length = new IntegerField();
    /**
     * Returns the length of the message.
     */
    public int getLength()
    {
        return _Length.getValue();
    }
    /**
     * Sets the length of the message.
     * This field is typically set by the AMPS client API rather than
     * by application code.
     */ 
    public void setLength(int v)
    {
        _Length.setValue(v);
    }

    StringField _Options = new StringField();
    /**
     * Returns the options set on this message.
     */
    public String getOptions()
    {
        return _Options.getValue(decoder);
    }
    /**
     * Sets the options for this message.
     * Options are a comma-delimited list of parameters. The values accepted
     * for options depend on the command: see the AMPS Command Reference for
     * details.
     */
    public void setOptions(String v)
    {
        _Options.setValue(v, encoder);
    }

    StringField _QueryId = new StringField();

    /**
     * Returns the QueryId for this message.
     * The QueryId returned on a message is the CommandId of the command
     * that ran the query. For example, when sending a SOW command to AMPS,
     * messages returned in response to that command will have the QueryId
     * set to the CommandId of the SOW command. 
     */
    public String getQueryId()
    {
        return _QueryId.getValue(decoder);
    }
    /**
     * Returns the QueryId for this message.
     * The QueryId returned on a message is the CommandId of the command
     * that ran the query. For example, when sending a SOW command to AMPS,
     * messages returned in response to that command will have the QueryId
     * set to the CommandId of the SOW command. 
     */    public Field getQueryIdRaw()
    {
        return _QueryId;
    }
    /**
     * Returns the QueryId for this message by copying it into the provided CommandId.
     * The QueryId returned on a message is the CommandId of the command
     * that ran the query. For example, when sending a SOW command to AMPS,
     * messages returned in response to that command will have the QueryId
     * set to the CommandId of the SOW command. 
     * @param v the CommandId to copy the QueryId into
     */    public boolean getQueryId(CommandId v)
    {
        return _QueryId.getValue(v);
    }
    /**
     * Sets the QueryId for this message.
     * Typically, the QueryId is set by the AMPS client on incoming messages.
     * @param v the value to set
     */
    public void setQueryId(String v)
    {
        _QueryId.setValue(v,encoder);
    }
    /**
     * Sets the QueryId for this message.
     * Typically, the QueryId is set for incoming messages by the AMPS client.
     * @param v the value to set
     */
    public void setQueryId(CommandId v)
    {
        _QueryId.setValue(v);
    }
    /**
     * Sets the QueryId for this message.
     * Typically, the QueryId is set by the AMPS client on incoming messages.
     * @param buffer the buffer containing the QueryId to set
     * @param offset the location where the QueryId begins
     * @param position the length of the QueryId to set
     */
    public void setQueryId(byte[] buffer,int offset,int position)
    {
        _QueryId.setValue(buffer,offset,position);
    }

    ReasonField _Reason = new ReasonField();
    /**
     * Returns the reason value of this message.
     * The reason is set on acknowledgement messages to provide more
     * information about the acknolwedgement.
     */
    public int getReason()
    {
        return _Reason.getValue();
    }
    /**
     * Returns the reason value of this message.
     * Typically, the reason is set for incoming messages by the AMPS client.
     */
    public String getReasonText()
    {
        return _Reason.getText();
    }
    /**
     * Sets the reason value of this message.
     * Typically, the reason is set for incoming messages by the AMPS client.
     */
     public void setReason(int v)
    {
        _Reason.setValue(v);
    }

    LongField _RecordsInserted = new LongField();
    /**
     * Returns the number of records inserted for the command that this message
     * was produced in response to.
    * This header is provided on an acknowledgement message that includes
    * information on the number of records inserted. See the AMPS Command
    * Reference for details.
     */
    public long getRecordsInserted()
    {
        return _RecordsInserted.getValue();
    }
    /**
     * Returns the number of records inserted for the command that this message
     * was produced in response to.
     * This header is provided on an acknowledgement message.
     * An application does not typically need to set this field.
     * See the AMPS Command Reference for details.
     */
    public void setRecordsInserted(long v)
    {
        _RecordsInserted.setValue(v);
    }

    LongField _RecordsUpdated = new LongField();
    /**
     * Returns the number of records updated for the command that this message
     * was produced in response to.
     * This header is provided on an acknowledgement message. See the
     * AMPS Command Reference for details.
     */
    public long getRecordsUpdated()
    {
        return _RecordsUpdated.getValue();
    }
    /**
     * Sets the records updated header field.
     * This header is provided on an acknowledgement message.
     * An application does not typically need to set this field.
     * See the AMPS Command Reference for details.
     */
    public void setRecordsUpdated(long v)
    {
        _RecordsUpdated.setValue(v);
    }

    LongField _RecordsDeleted = new LongField();

    /**
     * Returns the number of records deleted for the command that this message
     * was produced in response to.
     * This header is provided on an acknowledgement message. See the
     * AMPS Command Reference for details.
     */
    public long getRecordsDeleted()
    {
        return _RecordsDeleted.getValue();
    }
    /**
     * Sets the records deleted header field.
     * This header is provided on an acknowledgement message.
     * An application does not typically need to set this field.
     * See the AMPS Command Reference for details.
     */
    public void setRecordsDeleted(long v)
    {
        _RecordsDeleted.setValue(v);
    }

    LongField _RecordsReturned = new LongField();
    /**
     * Returns the number of records returned for the command that this message
     * was produced in response to.
     * This header is provided on an acknowledgement message. See the
     * AMPS Command Reference for details.
     */
    public long getRecordsReturned()
    {
        return _RecordsReturned.getValue();
    }
    /**
     * Sets the records returned header field.
     * This header is provided on an acknowledgement message.
     * An application does not typically need to set this field.
     * See the AMPS Command Reference for details.
     */
    public void setRecordsReturned(long v)
    {
        _RecordsReturned.setValue(v);
    }

    BooleanField _SendEmpties = new BooleanField();
    /**
     * Gets the value of the send empties header.
     * This header is one way of requesting that AMPS send empty
     * messages on a delta subscription. For new applications, using
     * the send_empties option is preferred.
     */
    public boolean getSendEmpties()
    {
        return _SendEmpties.getValue();
    }
    /**
     * Sets the value of the send empties header.
     * This header is one way of requesting that AMPS send empty
     * messages on a delta subscription. For new applications, using
     * the send_empties option is preferred.
     * @param v the new value
     */
    public void setSendEmpties(boolean v)
    {
        _SendEmpties.setValue(v);
    }

    BooleanField _SendMatchingIds = new BooleanField();
    /**
     * Gets the value of the send keys header.
     * This header is one way of requesting that AMPS send keys 
     * on a delta subscription. For new applications, using
     * the send_keys option is preferred.
     */
    public boolean getSendMatchingIds()
    {
        return _SendMatchingIds.getValue();
    }
    /**
     * Gets the value of the send keys header.
     * This header is one way of requesting that AMPS send keys 
     * on a delta subscription. For new applications, using
     * the send_keys option is preferred.
     * @param v the new value
     */
    public void setSendMatchingIds(boolean v)
    {
        _SendMatchingIds.setValue(v);
    }

    BooleanField _SendOOF = new BooleanField();
    /**
     * Gets the value of the send OOF header.
     * This header is one way of requesting that AMPS send OOF
     * messages for a SOW and subscribe. For new applications, using
     * the oof option is preferred.
     */
    public boolean getSendOOF()
    {
        return _SendOOF.getValue();
    }
    /**
     * Sets the value of the send OOF header.
     * This header is one way of requesting that AMPS send OOF
     * messages for a SOW and subscribe. For new applications, using
     * the oof option is preferred.
     * @param v the new value
     */
    public void setSendOOF(boolean v)
    {
        _SendOOF.setValue(v);
    }

    LongField _Sequence = new LongField();

    /**
     * Returns the sequence number for this message.
     * On outgoing messages, this field is typically set by the AMPS client.
     * The sequence number is used by AMPS for duplicate detection. Each
     * combination of client name and sequence number should be a unique
     * message. In the response to a login, AMPS may return the last sequence
     * number received from a client to assist in recovery.
     */
    public Field getSequenceRaw()
    {
        return _Sequence;
    }
    /**
     * Returns the sequence number for this message.
     * On outgoing messages, this field is typically set by the AMPS client.
     * The sequence number is used by AMPS for duplicate detection. Each
     * combination of client name and sequence number should be a unique
     * message. In response to a logon, AMPS may return the last sequence
     * number received from a client to assist in recovery.
     */
    public long getSequence()
    {
        return _Sequence.getValue();
    }
    /**
     * Returns the sequence number for this message.
     * This field is typically set by the AMPS client.
     * The AMPS client uses the sequence number in outgoing messages.
     * The sequence number is used by AMPS for duplicate detection. Each
     * combination of client name and sequence number should be a unique
     * message.
     */
    public void setSequence(long v)
    {
        _Sequence.setValue(v);
    }

    StringField _SowKey = new StringField();
    /**
     * Returns the SowKey for this message.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SowKey on each message.
     */
    public String getSowKey()
    {
        return _SowKey.getValue(decoder);
    }
    /**
     * Returns the SowKey for this message.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SowKey on each message.
     */
    public Field getSowKeyRaw()
    {
        return _SowKey;
    }
    /**
     * Sets the SowKey for this message.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     * @param v the new value for the SOW key
     */
    public void setSowKey(String v)
    {
        _SowKey.setValue(v,encoder);
    }
    /**
     * Sets the SowKey for this message.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     * @param buffer the buffer that contains the SOW key
     * @param offset the location within the buffer where the SOW key begins
     * @param the length of the SOW key 
     */
    public void setSowKey(byte[] buffer,int offset,int length)
    {
        _SowKey.setValue(buffer,offset,length);
    }

    StringField _SowKeys = new StringField();
   /**
     * Gets the set of SowKeys this message applies to.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     */
    public String getSowKeys()
    {
        return _SowKeys.getValue(decoder);
    }
   /**
     * Gets the set of SowKeys this message applies to.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     */
    public Field getSowKeysRaw()
    {
        return _SowKeys;
    }
   /**
     * Sets the set of SowKeys this message applies to, as a comma-delimited
     * list of identifiers.
     * This can be useful for commands that operate on multiple SOW records,
     * such as a sow_delete that specifies a set of keys to remove.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     * @param v the new value
     */
    public void setSowKeys(String v)
    {
        _SowKeys.setValue(v,encoder);
    }
   /**
     * Sets the set of SowKeys this message applies to, as a comma-delimited
     * list of identifiers.
     * This can be useful for commands that operate on multiple SOW records,
     * such as a sow_delete that specifies a set of keys to remove.
     * The SowKey is an opaque identifier used to uniquely identify a
     * SOW record within AMPS. For messages received from a SOW, AMPS provides
     * the SOW key on each message.
     * @param buffer the buffer that contains the SOW key
     * @param offset the location within the buffer where the SOW key begins
     * @param length the length of the SOW key
     */
    public void setSowKeys(byte[] buffer,int offset,int length)

    {
        _SowKeys.setValue(buffer,offset,length);
    }

    StatusField _Status = new StatusField();
   /**
    * Returns the status of this message. This field is provided by AMPS
    * on acknowledgements to indicate the success or failure of the
    * command being acknowledged.
    */
    public int getStatus()
    {
        return _Status.getValue();
    }
   /**
    * Sets the status of this message. This field is provided by AMPS
    * on acknowledgements to indicate the success or failure of the
    * command being acknowledged.
    */
    public void setStatus(int v)
    {
        _Status.setValue(v);
    }

    StringField _SubId = new StringField();
    /**
     * Gets the SubId of this message.
     * The SubId is a reference to the CommandId of the command that
     * created the subscription.
     */
    public String getSubId()
    {
        return _SubId.getValue(decoder);
    }
    /**
     * Gets the SubId of this message.
     * The SubId is a reference to the CommandId of the command that
     * created the subscription.
     */
    public Field getSubIdRaw()
    {
        return _SubId;
    }
    /**
     * Gets the SubId of this message by copying it into the provided CommandId.
     * The SubId is a reference to the CommandId of the command that
     * created the subscription.
     */
    public boolean getSubId(CommandId v)
    {
        return _SubId.getValue(v);
    }
    /**
     * Sets the SubId of this message.
     * The SubId is a reference to the CommandId of the command that
     * created the subscription.
     */
    public void setSubId(String v)
    {
        _SubId.setValue(v,encoder);
    }
    /**
     * Sets the SubId of this message.
     * The SubId is the CommandId of the command that
     * created the subscription. For example, to replace a subscription,
     * you provide the SubId of the subscription to be replaced with the
     * command that replaces the subscription.
     * @param v the new value of the SubId
     */
    public void setSubId(CommandId v)
    {
        _SubId.setValue(v);
    }
    /**
     * Sets the SubId of this message.
     * The SubId is a reference to the CommandId of the command that
     * created the subscription. For example, messages returned for a
     * subscription contain the SubId of the subscription.
     * @param buffer the buffer that contains the SubId key
     * @param offset the location within the buffer where the SubId begins
     * @param the length of the SubId
     */
    public void setSubId(byte[] buffer,int offset,int length)
    {
        _SubId.setValue(buffer,offset,length);
    }

    StringField _SubIds = new StringField();
    /**
     * The set of SubIds for this message. AMPS returns a SubId for each
     * subscription that matches the message as a comma-delimited list.
     * The SubId is the identifier provided when the subscription was
     * registered with AMPS. 
     */
    public String getSubIds()
    {
        return _SubIds.getValue(decoder);
    }
    /**
     * The set of SubIds for this message. AMPS returns a SubId for each
     * subscription that matches the message as a comma-delimited list.
     * The SubId is the identifier provided when the subscription was
     * registered with AMPS. 
     */
    public Field getSubIdsRaw()
    {
        return _SubIds;
    }
    /**
     * The set of SubIds for this message. AMPS returns a SubId for each
     * subscription that matches the message as a comma-delimited list.
     * The SubId is the identifier provided when the subscription was
     * registered with AMPS. 
     */
    public void setSubIds(String v)
    {
        _SubIds.setValue(v,encoder);
    }
    /**
     * The set of SubIds for this message. AMPS returns a SubId for each
     * subscription that matches the message.
     * The SubId is the identifier provided when the subscription was
     * registered with AMPS. 
     */
    public void setSubIds(byte[] buffer,int offset,int length)
    {
        _SubIds.setValue(buffer,offset,length);
    }

    StringField _Timestamp = new StringField();
    /**
     * Returns the timestamp for this message, an ISO-8601 formatted string.
     * The timestamp is set by AMPS at the time that AMPS processes the message.
     */
    public String getTimestamp()
    {
        return _Timestamp.getValue(decoder);
    }
    /**
     * Returns the timestamp for this message, an ISO-8601 formatted string.
     * The timestamp is set by AMPS  at the time that AMPS processes
     * the message.
     */
    public Field getTimestampRaw()
    {
        return _Timestamp;
    }
    /**
     * Sets the timestamp for this message. The timestamp is set by AMPS
     * at the time that AMPS processes the message. There is generally no
     * need to set this field in your application.
     * @param v the timestamp to set
     */
    public void setTimestamp(String v)
    {
        _Timestamp.setValue(v,encoder);
    }
    /**
     * Sets the timestamp for this message. The timestamp is set by AMPS
     * at the time that AMPS processes the message. There is generally no
     * need to set this field in your application.
     * @param buffer the buffer that contains the timestamp
     * @param offset the location within the buffer where the timestamp begins
     * @param length the length of the timestamp 
     */
    public void setTimestamp(byte[] buffer,int offset,int length)
    {
        _Timestamp.setValue(buffer,offset,length);
    }

    LongField _TopN = new LongField();
   /**
    * Get the TopN parameter for this message.
    * For commands that support limiting the number of messages in the result
    * set, this parameter provides the limit to AMPS.
    */
    public long getTopN()
    {
        return _TopN.getValue();
    }
   /**
    * Set the TopN parameter for this message.
    * For commands that support limiting the number of messages in the result
    * set, this parameter provides the limit to AMPS.
    * @param v the value to set
    */
    public void setTopN(long v)
    {
        _TopN.setValue(v);
    }

    StringField _Topic = new StringField();
    /**
     * Return the topic that the message applies to.
     */
    public String getTopic()
    {
        return _Topic.getValue(decoder);
    }
    /**
     * Return the topic that the message applies to.
     */
    public Field getTopicRaw()
    {
        return _Topic;
    }
    /**
     * Set the topic that the message applies to.
     * @param v the topic to set
     */
    public void setTopic(String v)
    {
        _Topic.setValue(v,encoder);
    }
    /**
     * Set the topic that the message applies to.
     * @param buffer the buffer that contains the timestamp
     * @param offset the location within the buffer where the timestamp begins
     * @param length the length of the timestamp 
     */
    public void setTopic(byte[] buffer,int offset,int length)
    {
        _Topic.setValue(buffer,offset,length);
    }

    LongField _TopicMatches = new LongField();
   /**
     * Returns the number of matching topics for the command that this message
     * was produced in response to.
     * This header is provided on an acknowledgement message that includes
     * information on the number of records inserted. See the AMPS Command
     * Reference for details.
     */ 
    public long getTopicMatches()
    {
        return _TopicMatches.getValue();
    }

    /**
     * Sets the topic matches field on this message. This header is typically
     * provided by AMPS.
     * @param v the value to set
     */
    public void setTopicMatches(long v)
    {
        _TopicMatches.setValue(v);
    }

    StringField _UserId = new StringField();
    /**
     * Get the UserId for this message. This field is typically set during
     * the logon sequence. The field may also be provided on published
     * messages, depending on the authentication used in AMPS. When provided
     * on a published message, AMPS provides the identity of the connection
     * that sent the message, not the value provided with the message
     * as published.
     */
    public String getUserId()
    {
        return _UserId.getValue(decoder);
    }

    /**
     * Get the UserId for this message. This field is typically set during
     * the logon sequence. The field may also be provided on published
     * messages, depending on the authentication used in AMPS. When provided
     * on a published message, AMPS provides the identity of the connection
     * that sent the message, not the value provided with the message
     * as published.
     */

    public Field getUserIdRaw()
    {
        return _UserId;
    }

    /**
     * Set the UserId for this message. This field is typically set during
     * the logon sequence. The field may also be provided on published
     * messages, depending on the authentication used in AMPS. When provided
     * on a published message, AMPS provides the identity of the connection
     * that sent the message, not the value provided with the message
     * as published. Therefore, applications generally only set this
     * value during logon.
     * @param v the UserId to set
     */
    public void setUserId(String v)
    {
        _UserId.setValue(v,encoder);
    }
    /**
     * Set the UserId for this message. This field is typically set during
     * the logon sequence. The field may also be provided on published
     * messages, depending on the authentication used in AMPS. When provided
     * on a published message, AMPS provides the identity of the connection
     * that sent the message, not the value provided with the message
     * as published. Therefore, applications generally only set this
     * value during logon.
     * @param buffer the buffer that contains the UserId
     * @param offset the position where the UserId begins
     * @param length the length of the UserId
     */

    public void setUserId(byte[] buffer,int offset,int length)
    {
        _UserId.setValue(buffer,offset,length);
    }

    StringField _Version = new StringField();

   /**
    * Returns the version of the AMPS server, provided in response to a logon.
    */
    public String getVersion()
    {
        return _Version.getValue(decoder);
    }
    /**
     * Returns the version of the AMPS server, provided in response to a logon.
     */
    public Field getVersionRaw()
    {
        return _Version;
    }
    /**
     * Returns the version of the AMPS server, provided in response to a logon.
     */
    public int getVersionAsInt()
    {
        // No version is 0
        if (_Version.isNull()) return 0;

        int version = 0;
        try
        {
            version = Client.getVersionAsInt(_Version.getValue(decoder));
        }
        catch (CommandException e) {;}
        if (version == 0)
            version = MINIMUM_SERVER_VERSION;

        return version;
    }
    /**
     * Sets the version field of this message. This is typically done by
     * the AMPS client API when a message with this field set is received.
     * @param v the version to set
     */
    public void setVersion(String v)
    {
        _Version.setValue(v,encoder);
    }
    /**
     * Sets the version field of this message. This is typically done by
     * the AMPS client API when a message with this field set is received.
     * @param buffer the buffer that contains the version string
     * @param offset the position where the version string begins
     * @param length the length of the version string
     */
     
    public void setVersion(byte[] buffer,int offset,int length)
    {
        _Version.setValue(buffer,offset,length);
    }

    StringField _CorrelationId = new StringField();
    /**
     * Set the CorrelationId on this message. The CorrelationId
     * is an opaque identifier provided by the publisher of a message.
     * AMPS provides the identifier to subscribers of the message without
     * interpreting or changing the identifier.
     * @param v the value to set
     */ 
    public void setCorrelationId(String v)
    {
        _CorrelationId.setValue(v,encoder);
    }
    /**
     * Get the CorrelationId on this message. The CorrelationId
     * is an opaque identifier provided by the publisher of a message.
     * AMPS provides the identifier to subscribers of the message without
     * interpreting or changing the identifier.
     */ 
    public String getCorrelationId()
    {
        return _CorrelationId.getValue(decoder);
    }
    /**
     * Get the CorrelationId on this message. The CorrelationId
     * is an opaque identifier provided by the publisher of a message.
     * AMPS provides the identifier to subscribers of the message without
     * interpreting or changing the identifier.
     */
     public StringField getCorrelationIdRaw()
    {
        return _CorrelationId;
    }

    /**
     * Return the message as a string. This serializes the message and returns
     * the serialized value.
     */
    public String toString()
    {
        String tmp = "";
        StringBuilder s = new StringBuilder(1024);
        s.append("Message{");

        s.append("Command=").append(CommandField.encodeCommand(getCommand())).append("; ");
        try
        {
            // IDEA:
            //    Use try/catch coupled with a string to fail absolutely and not
            //    append anything if the Field throws a NullPointerEx when evaluating
            //    if the buffer is null.
            tmp = "AckType=" + _AckType.toString();
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }
        try
        {
            tmp = "BatchSize=" + getBatchSize();
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getClientName() != null)
            s.append("ClientName=").append(getClientName()).append("; ");

        if(getCommandId()  != null)
            s.append("CommandId=").append(getCommandId()).append("; ");

        try
        {
            tmp = "Expiration=" + getExpiration();
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if ( getFilter() != null)
            s.append("Filter=").append(getFilter()).append("; ");

        if ( getOrderBy() != null)
            s.append("OrderBy=").append(getOrderBy()).append("; ");

        try
        {
            tmp = "GroupSeqNo=" + Long.toString(getGroupSeqNo());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "Matches=" + Long.toString(getMatches());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "MaxMessages=" + Long.toString(getMaxMessages());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getMessageId()  != null)
            s.append("MessageId=").append(getMessageId()).append("; ");

        try
        {
            tmp = "Length=" + Integer.toString(getLength());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "Options=" + getOptions();
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getPassword()   != null)
            s.append("Password=").append(getPassword()).append("; ");

        if(getQueryId()    != null)
            s.append("QueryId=").append(getQueryId()).append("; ");

        try
        {
            tmp = "Reason=" + ReasonField.encodeReason(getReason());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "RecordsUpdated=" + Long.toString(getRecordsUpdated());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "RecordsInserted=" + Long.toString(getRecordsInserted());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "RecordsReturned=" + Long.toString(getRecordsReturned());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "RecordsDeleted=" + Long.toString(getRecordsDeleted());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "SendEmpties=" + (getSendEmpties() ? "true" : "false");
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "SendMatchingIds=" + (getSendMatchingIds() ? "true" : "false");
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        try
        {
            tmp = "SendOOF=" + (getSendOOF() ? "true" : "false");
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getSowKey()     != null)
            s.append("SowKey=").append(getSowKey()).append("; ");

        if(getSowKeys()    != null)
            s.append("SowKeys=").append(getSowKeys()).append("; ");

        try
        {
            tmp = "Status=" + StatusField.encodeStatus(getStatus());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getSubId()      != null)
            s.append("SubId=").append(getSubId()).append("; ");

        if(getSubIds()     != null)
            s.append("SubIds=").append(getSubIds()).append("; ");

        if(getTimestamp()  != null)
            s.append("Timestamp=").append(getTimestamp()).append("; ");

        try
        {
            tmp = "TopN=" + Long.toString(getTopN());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }

        if(getTopic()      != null)
            s.append("Topic=").append(getTopic()).append("; ");

        try
        {
            tmp = "TopicMatches=" + Long.toString(getTopicMatches());
            s.append(tmp + "; ");
        }
        catch (java.lang.NullPointerException npe)
        {
            ;
        }
        String correlationId = getCorrelationId();
        if(correlationId != null) s.append("CorrelationId=").append(correlationId).append("; ");
        
        if(getUserId()     != null)
            s.append("UserId=").append(getUserId().toString()).append("; ");

        if(getData()       != null)
            s.append("Data=").append(getData()).append("; ");

        s.append("}");

        return s.toString();
    }
    public abstract Message copy();
    
    protected void _copyTo(Message destination)
    {
        destination.buffer = new byte[this.buffer.length];
        destination._rawBufferLength = _rawBufferLength;
        destination._rawBufferOffset = _rawBufferOffset;
        System.arraycopy(buffer, 0, destination.buffer, 0, buffer.length);
        destination._AckType.set(destination.buffer, _AckType.position, _AckType.length);
        destination._BatchSize.set(destination.buffer, _BatchSize.position, _BatchSize.length);
        destination._Bookmark.set(destination.buffer, _Bookmark.position, _Bookmark.length);
        destination._ClientName.set(destination.buffer, _ClientName.position, _ClientName.length);
        destination._Command.set(destination.buffer, _Command.position, _Command.length);
        destination._CommandId.set(destination.buffer, _CommandId.position, _CommandId.length);
        destination._CorrelationId.set(destination.buffer, _CorrelationId.position, _CorrelationId.length);
        destination._Data.set(destination.buffer, _Data.position, _Data.length);
        destination._Expiration.set(destination.buffer, _Expiration.position, _Expiration.length);
        destination._Filter.set(destination.buffer, _Filter.position, _Filter.length);
        destination._GroupSeqNo.set(destination.buffer, _GroupSeqNo.position, _GroupSeqNo.length);
        destination._Length.set(destination.buffer, _Length.position, _Length.length);
        destination._Matches.set(destination.buffer, _Matches.position, _Matches.length);
        destination._MaxMessages.set(destination.buffer, _MaxMessages.position, _MaxMessages.length);
        destination._MessageId.set(destination.buffer, _MessageId.position, _MessageId.length);
        destination._Options.set(destination.buffer, _Options.position, _Options.length);
        destination._Password.set(destination.buffer, _Password.position, _Password.length);
        destination._QueryId.set(destination.buffer, _QueryId.position, _QueryId.length);
        destination._Reason.set(destination.buffer, _Reason.position, _Reason.length);
        destination._RecordsInserted.set(destination.buffer, _RecordsInserted.position, _RecordsInserted.length);
        destination._RecordsUpdated.set(destination.buffer, _RecordsUpdated.position, _RecordsUpdated.length);
        destination._RecordsDeleted.set(destination.buffer, _RecordsDeleted.position, _RecordsDeleted.length);
        destination._RecordsReturned.set(destination.buffer, _RecordsReturned.position, _RecordsReturned.length);
        destination._SendMatchingIds.set(destination.buffer, _SendMatchingIds.position, _SendMatchingIds.length);
        destination._SendOOF.set(destination.buffer, _SendOOF.position, _SendOOF.length);
        destination._Sequence.set(destination.buffer, _Sequence.position, _Sequence.length);
        destination._SowKey.set(destination.buffer, _SowKey.position, _SowKey.length);
        destination._SowKeys.set(destination.buffer, _SowKeys.position, _SowKeys.length);
        destination._Status.set(destination.buffer, _Status.position, _Status.length);
        destination._SubId.set(destination.buffer, _SubId.position, _SubId.length);
        destination._SubIds.set(destination.buffer, _SubIds.position, _SubIds.length);
        destination._Timestamp.set(destination.buffer, _Timestamp.position, _Timestamp.length);
        destination._Topic.set(destination.buffer, _Topic.position, _Topic.length);
        destination._TopicMatches.set(destination.buffer, _TopicMatches.position, _TopicMatches.length);
        destination._TopN.set(destination.buffer, _TopN.position, _TopN.length);
        destination._UserId.set(destination.buffer, _UserId.position, _UserId.length);
        destination._Version.set(destination.buffer, _Version.position, _Version.length);
        destination._bookmarkSeqNo = _bookmarkSeqNo;
    }
    
}

