////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import com.crankuptheamps.client.exception.CommandException;

public class NVFIXBuilder
{
    private byte[] buffer   = null;
    private int    size     = 0;
    private int    capacity = 1024;

    private byte   fieldSeparator;

    public NVFIXBuilder(int capacity, byte fieldSeparator)
    {
        this.fieldSeparator = fieldSeparator;
        this.capacity = capacity;
        this.buffer = new byte[capacity];
    }

    /**
     * Clears the current builder, resetting it to empty.
     */
    public void clear()
    {
        size = 0;
    }
    
    public int getSize()
    {
        return this.size;
    }

    public byte[] getBytes()
    {
        return buffer;
    }

    private void checkCapacity(int bytesNeeded)
    {
        if(capacity - size < bytesNeeded)
        {
            // Not enough capacity left, must resize
            while (capacity - size < bytesNeeded)
            {
                capacity = capacity * 2;
            }

            byte[] newbuf = new byte[capacity];
            System.arraycopy(buffer, 0, newbuf, 0, size);

            buffer = newbuf;
        }
    }

    /**
     * Appends a tag and value pair contained within a byte buffer to the FIX message.
     *
     * @param tag
     *          the String NVFIX tag to append
     * @param value
     *          the byte buffer containing the FIX value to append.
     * @param offset
     *          the starting location of the value inside the byte buffer.
     * @param length
     *          the length of the value inside the byte buffer.
     * @throws CommandException
     *          if the tag or value argument is not convertible to ISO-8859-1.
     * @return a reference to this object.
     */
    public NVFIXBuilder append(String tag, byte[] value, int offset, int length)
    throws CommandException
    {
        byte[] tagBytes;

        try
        {
            tagBytes = tag.getBytes("ISO-8859-1");
        }
        catch (java.io.UnsupportedEncodingException ioex)
        {
            throw new CommandException("Unable to encode tag '" + tag
                                       + "' to a byte array in FIXBuilder.\n" + ioex.toString());
        }
        int sizeNeeded = tagBytes.length + length + 2;
        checkCapacity(sizeNeeded);

        // Tag
        int writeIndex = size;
        System.arraycopy(tagBytes, 0, buffer, writeIndex, tagBytes.length);
        writeIndex += tagBytes.length;

        // Equal sign
        buffer[writeIndex] = '=';
        ++writeIndex;

        // Value
        System.arraycopy(value, offset, buffer, writeIndex, length);
        writeIndex += length;

        // Field separator
        buffer[writeIndex] = fieldSeparator;
        size += sizeNeeded;

        return this;
    }


    /**
     * Appends a tag and value pair to the FIX message.
     *
     * @param tag
     *           the string FIX tag to append
     * @param value
     *           the FIX value for the tag to append. The value will be converted
     *           to an ISO-8859-1 byte array for writing.
     * @throws CommandException
     *            if the tag or value argument is not convertible to ISO-8859-1.
     * @return a reference to this object.
     */
    public NVFIXBuilder append(String tag, String value) throws CommandException
    {
        byte[] tagBytes;
        byte[] valueBytes;
        try
        {
            tagBytes = tag.getBytes("ISO-8859-1");
        }
        catch (java.io.UnsupportedEncodingException ioex)
        {
            throw new CommandException("Unable to encode tag '" + tag
                                       + "' to a byte array in FIXBuilder.\n" + ioex.toString());
        }
        try
        {
            valueBytes = value.getBytes("ISO-8859-1");
        }
        catch (java.io.UnsupportedEncodingException ioex)
        {
            throw new CommandException("Unable to encode value '" + value
                                       + "' to a byte array in FIXBuilder.\n" + ioex.toString());
        }
        int sizeNeeded = tagBytes.length + valueBytes.length + 2;
        checkCapacity(sizeNeeded);

        // Tag
        int writeIndex = size;
        System.arraycopy(tagBytes, 0, buffer, writeIndex, tagBytes.length);
        writeIndex += tagBytes.length;

        // Equal sign
        buffer[writeIndex] = '=';
        ++writeIndex;

        // Value
        System.arraycopy(valueBytes, 0, buffer, writeIndex, valueBytes.length);
        writeIndex += valueBytes.length;

        // Field separator
        buffer[writeIndex] = fieldSeparator;
        size += sizeNeeded;

        return this;
    }

}
