////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;
import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.StoreException;
import com.crankuptheamps.client.exception.TimedOutException;

/**
 * Represents a message store. The AMPS client uses message stores for
 * recovery purposes. The store is responsible for maintaining the state
 * of published messages and recovering that state in the event of a
 * disconnection. Optionally, the store may persist message state and
 * recover that state if the application restarts.
 *
 */
public interface Store
{

   /**
    * Replay the messages saved in a store. Provided to the
    * {@link Store#replay} or {@link Store#replaySingle} methods
    * to replay stored operations. The StoreReplayer is responsible for
    * publishing the operations to AMPS.
    */
    
    public interface StoreReplayer
    {
        /**
         * Replay the specified operation.
         * @param index an index that identifies this operation
         * @param operation the operation to replay
         * @param topic a byte array containing the name of the topic for this operation
         * @param topicOffset the offset in the topic array where the topic name begins
         * @param topicLength the length of the topic name
         * @param data a byte array containing the data for this operation
         * @param dataOffset the offset into the data array where the data begins
         * @param dataLength the length of the data
         * @param correlationId the correlation ID for this message
         * @param correlationIdOffset the offset into the correlation ID array where the correlation ID begins
         * @param correlationIdLength the length of the correlation ID
         */
    
        public void execute(long index, int operation, byte[] topic,
                            long topicOffset, long topicLength, byte[] data,
                            long dataOffset, long dataLength,
                            byte[] correlationId, long correlationIdOffset, long correlationIdLength)
                                throws DisconnectedException;

        /**
         * Replay the specified operation.
         * @param index an index that identifies this operation
         * @param operation the operation to replay
         * @param topic a byte array containing the name of the topic for this operation
         * @param topicOffset the offset in the topic array where the topic name begins
         * @param topicLength the length of the topic name
         * @param data a byte array containing the data for this operation
         * @param dataOffset the offset into the data array where the data begins
         * @param dataLength the length of the data
         * @param correlationId the correlation ID for this message
         * @param correlationIdOffset the offset into the correlation ID array where the correlation ID begins
         * @param correlationIdLength the length of the correlation ID
         * @param expiration the expiration to set on the operation
         */

        public void execute(long index, int operation, byte[] topic,
                            long topicOffset, long topicLength, byte[] data,
                            long dataOffset, long dataLength,
                            byte[] correlationId, long correlationIdOffset, long correlationIdLength,
                            int expiration)
                                throws DisconnectedException;

        /**
         * Replay the specified operation.
         * @param index an index that identifies this operation
         * @param operation the operation to replay
         * @param topic a byte array containing the name of the topic for this operation
         * @param topicOffset the offset in the topic array where the topic name begins
         * @param topicLength the length of the topic name
         * @param data a byte array containing the data for this operation
         * @param dataOffset the offset into the data array where the data begins
         * @param dataLength the length of the data
         * @param expiration the expiration to set on the operation
         * @param cmdId the commandID to set on the operation
         */
        public void execute(long index, int operation, byte[] topic,
                            long topicOffset, long topicLength, byte[] data,
                            long dataOffset, long dataLength, int expiration,
                            CommandId cmdId)
                                throws DisconnectedException;
    }

    public static final int SOWDeleteByData = 1;
    public static final int SOWDeleteByFilter = 2;
    public static final int SOWDeleteByKeys = 4;


    /**
     * Store the provided operation.  The index provided is a key that
     * the client can later use to replay the operation or remove the
     * operation from the store. Implementations may assume that the index
     * increases monotonically.
     *
     * @param index an index that identifies this operation
     * @param operation the operation to store
     * @param topic a byte array containing the name of the topic for this operation
     * @param topicOffset the offset in the topic array where the topic name begins
     * @param topicLength the length of the topic name
     * @param data a byte array containing the data for this operation
     * @param dataOffset the offset into the data array where the data begins
     * @param dataLength the length of the data
     * @param correlationId the correlation ID for this message
     * @param correlationIdOffset the offset into the correlation ID array where the correlation ID begins
     * @param correlationIdLength the length of the correlation ID
     */
    public void store(long index, int operation, byte[] topic,
                      long topicOffset, long topicLength, byte[] data,
                      long dataOffset, long dataLength,
                      byte[] correlationId, long correlationIdOffset, long correlationIdLength) throws StoreException;
    /**
     * Store the provided operation.  The index provided is a key that
     * the client can later use to replay the operation or remove the
     * operation from the store. Implementations may assume that the index
     * increases monotonically.
     *
     * @param index an index that identifies this operation
     * @param operation the operation to store
     * @param topic a byte array containing the name of the topic for this operation
     * @param topicOffset the offset in the topic array where the topic name begins
     * @param topicLength the length of the topic name
     * @param data a byte array containing the data for this operation
     * @param dataOffset the offset into the data array where the data begins
     * @param dataLength the length of the data
     * @param expiration the expiration to set on the operation
     * @param correlationId the correlation ID for this message
     * @param correlationIdOffset the offset into the correlation ID array where the correlation ID begins
     * @param correlationIdLength the length of the correlation ID
     */
    public void store(long index, int operation, byte[] topic,
                      long topicOffset, long topicLength, byte[] data,
                      long dataOffset, long dataLength, 
                      byte[] correlationId, long correlationIdOffset, long correlationIdLength,
                      int expiration)
                            throws StoreException;
    /**
     * Store the provided operation.  The index provided is a key that
     * the client can later use to replay the operation or remove the
     * operation from the store. Implementations may assume that the index
     * increases monotonically.
     *
     * @param index an index that identifies this operation
     * @param operation the operation to store
     * @param topic a byte array containing the name of the topic for this operation
     * @param topicOffset the offset in the topic array where the topic name begins
     * @param topicLength the length of the topic name
     * @param data a byte array containing the data for this operation
     * @param dataOffset the offset into the data array where the data begins
     * @param dataLength the length of the data
     * @param expiration the expiration to set on the operation
     * @param cmdId the commandId to set on the operation
     */
    public void store(long index, int operation, byte[] topic,
                      long topicOffset, long topicLength, byte[] data,
                      long dataOffset, long dataLength, int expiration, CommandId cmdId)
                            throws StoreException;

    
    /**
     * Discard all operations up to the index provided.
     * @param index the index number to keep -- all previous index numbers will be discarded
     */
     
    public void discardUpTo(long index) throws StoreException;

    /**
     * Replay all operations in the store using the provided StoreReplayer.
     * @param replayer the StoreReplayer to use to replay the operations
     */

    public void replay(StoreReplayer replayer) throws StoreException, DisconnectedException;

    /**
     * Replay the operation at the specified index.
     */
    
    public void replaySingle(StoreReplayer replayer, long index) throws StoreException, DisconnectedException;
    
    /**
     * Return the number of operations in the store.
     */
    public long unpersistedCount();

    /**
     * Return the oldest index in the store.
     */
    public long getLowestUnpersisted();

    /**
     * Wait for the store to discard everything that has been stored
     * up to the point in time when flush is called, then
     * return.
     */
    public void flush() throws TimedOutException;

    /**
     * Wait for the store to discard everything that has been stored
     * up to the point in time when flush is called, then
     * return. Throw an exception if this is not completed in the number of
     * milliseconds specified by the timeout.
     * @param timeout the number of milliseconds to wait for the flush
     */
    public void flush(long timeout) throws TimedOutException;

    /**
     * Set the resize handler for the publish store. An implementation can
     * call the resize handler in the event that it needs to grow the
     * size of the publish store.
     */
    public void setResizeHandler(PublishStoreResizeHandler handler);
}
