////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.Socket;
import java.net.SocketException;
import java.net.URI;
import java.beans.ExceptionListener;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedByInterruptException;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.TimeUnit;
import java.util.Iterator;
import java.util.Map;
import java.util.Properties;

import com.crankuptheamps.client.exception.AlreadyConnectedException;
import com.crankuptheamps.client.exception.ConnectionRefusedException;
import com.crankuptheamps.client.exception.DisconnectedException;
import com.crankuptheamps.client.exception.RetryOperationException;
import com.crankuptheamps.client.exception.InvalidURIException;

public class TCPTransportImpl
{
    private   URI               _addr;
    protected SocketChannel     _socket            = null;
    //public final Lock         _lock              = new DebugLock("Impl");
    public    final Lock         _lock              = new ReentrantLock();

    volatile int _connectionVersion = 0;
    private volatile boolean _disconnecting = false;

    private Protocol       _messageType       = null;
    private MessageHandler    _onMessage         = DefaultMessageHandler.instance;
    private TransportDisconnectHandler _onDisconnect = DefaultDisconnectHandler.instance;
    private TCPReaderThread   _readerThread      = null;
    private ExceptionListener _exceptionListener = null;
    private Properties        _properties = null;
    private Selector          _selector = null;
    private int _readTimeout = 0;
    protected TransportFilter _filter;

    public TCPTransportImpl(Protocol messageType, Properties properties, TransportFilter filter)
    {
        this._messageType   = messageType;
        this._properties    = properties;
        this._filter = filter;
    }

    public void setMessageHandler(MessageHandler h)
    {
        this._onMessage = h;
    }

    public void setDisconnectHandler(TransportDisconnectHandler h)
    {
        this._onDisconnect = h;
    }

    public void setExceptionListener(ExceptionListener exceptionListener)
    {
        this._exceptionListener = exceptionListener;
    }

    public void setTransportFilter(TransportFilter filter)
    {
        this._filter = filter;
    }
    public void connect(URI addr) throws ConnectionRefusedException, AlreadyConnectedException, InvalidURIException
    {
        _lock.lock();
        _disconnecting = false;
        // clear out the interrupt bit.  If we don't do this, and the thread is interrupted,
        // EVERY SUBSEQUENT CALL TO connect() WILL FAIL FOREVER FROM THIS THREAD and throw a
        // ClosedByInterruptException.  Unlike InterruptedException, ClosedByInterruptException
        // does not clear the thread's interrupted state.
        Thread.interrupted();
        try
        {
            if(this._addr != null)
            {
                throw new AlreadyConnectedException("Already connected to AMPS at " +
                                                    this._addr.getHost() + ":" + this._addr.getPort() + "\n");
            }
            _socket = createSocket();

            // Merge properties from construction and the ones in this URI.
            URIProperties properties = new URIProperties(addr);
            if(_properties != null) properties.putAll(_properties);
            applySocketProperties(properties);

            _socket.connect(new InetSocketAddress(addr.getHost(), addr.getPort()));
            while(!_socket.finishConnect())
            {
                Thread.yield();
            }
            if(this._selector != null)
            {
                this._selector.close();
            }

            try
            {
                this._selector = Selector.open();
            } catch(IOException ioex)
            {
                // no selector available.  Client-side heartbeat detection is disabled.
                this._selector = null;
                ioex.printStackTrace();
            }
            _readerThread = new TCPReaderThread(this, this._messageType);
            _addr = addr;
            _connectionVersion++;
        }
        catch (ClosedByInterruptException e)
        {
            // Clear the interrupt flag!  It is set, and simply catching this exception
            // doesn't clear it.
            Thread.interrupted();
            throw new ConnectionRefusedException("Interrupted, but please try again.", e);

        }
        catch (IOException ioex)
        {
            throw new ConnectionRefusedException("Unable to connect to AMPS at " +
                                                 addr.getHost() + ":" + addr.getPort() , ioex);
        }
        catch (IllegalArgumentException iaex)
        {
            throw new InvalidURIException("Error setting socket options", iaex);
        }
        finally
        {
            _lock.unlock();
        }
    }

    protected SocketChannel createSocket() throws IOException
    {
        SocketChannel socket = SocketChannel.open();
        socket.configureBlocking(false);
        return socket;
    }

    private void applySocketProperties(Properties properties_) throws SocketException, InvalidURIException
    {
        Socket s = _socket.socket();
        s.setKeepAlive(true);
        // We default SO_LINGER on, on Java, because
        // a shutdown() option isn't available on Java SocketChannels until JDK 1.7.
        s.setSoLinger(true, 10);
        if(properties_ == null) return;

        for(Map.Entry<Object, Object> entry : properties_.entrySet())
        {
            Object key = entry.getKey();
            Object value = entry.getValue();
            if("tcp_keepalive".equals(key))
            {
                if("false".equals(value))
                {
                    s.setKeepAlive(false);
                }
                else if("true".equals(value))
                {
                    s.setKeepAlive(true);
                }
                else throw new InvalidURIException("Invalid value for tcp_keepalive.");
            }
            else if("tcp_sndbuf".equals(key))
            {
                try
                {
                    int sndbuf = Integer.parseInt((String)value);
                    s.setSendBufferSize(sndbuf);
                }
                catch(NumberFormatException ex)
                {
                    throw new InvalidURIException("Invalid value for tcp_sndbuf.");
                }
            }
            else if("tcp_rcvbuf".equals(key))
            {
                try
                {
                    int rcvbuf = Integer.parseInt((String)value);
                    s.setReceiveBufferSize(rcvbuf);
                }
                catch(NumberFormatException ex)
                {
                    throw new InvalidURIException("Invalid value for tcp_rcvbuf.");
                }
            }
            else if("tcp_linger".equals(key))
            {
                try
                {
                    int linger = Integer.parseInt((String)value);
                    if(linger == -1)
                    {
                        s.setSoLinger(false, 0);
                    }
                    else
                    {
                        s.setSoLinger(true, linger);
                    }
                }
                catch(NumberFormatException ex)
                {
                    throw new InvalidURIException("Invalid value for tcp_linger.");
                }
            }
            else if("tcp_nodelay".equals(key))
            {
                if("false".equals(value))
                {
                    s.setTcpNoDelay(false);
                }
                else if ("true".equals(value))
                {
                    s.setTcpNoDelay(true);
                }
                else throw new InvalidURIException("Invalid value for tcp_nodelay.");
            }
            else
            {
                throw new InvalidURIException("Unrecognized URI parameter `" + key +"'");
            }
        }
    }


    private void _disconnect()
    {
        try
        {
            if(_addr != null)
            {
                _readerThread.stopThread();
                _socket.close();

                // we'd like the next blocking operation in the
                // reader thread to throw so it can end.  Unless,
                // of course, *we* are the reader thread.
                if(!_readerThread.equals(Thread.currentThread()))
                {
                    _readerThread.interrupt();
                    _readerThread.join();
                }
                _selector.close();
            }
        }
        catch (Exception ex)
        {
            ; // should be no harm absorbing
        }
        this._addr = null;
    }

    public void disconnect()
    {
        _lock.lock();
        try
        {
            _disconnecting = true;
            _disconnect();
        }
        finally
        {
            _lock.unlock();
        }
    }

    public void send(ByteBuffer buf) throws DisconnectedException
    {
        try
        {
            _filter.outgoing(buf);
            while(buf.hasRemaining())
            {
                _socket.write(buf);
            }
        }
        catch (NullPointerException ioex)
        {
            throw new DisconnectedException("Socket error while sending message.");
        }
        catch (IOException ioex)
        {
            throw new DisconnectedException("Socket error while sending message.");
        }
    }


    public Socket socket()
    {
        try
        {
            return _socket.socket();
        }
        catch(Exception e)
        {
            return null;
        }
    }

    public long writeQueueSize()
    {
        return 0;
    }

    public long readQueueSize()
    {
        return 0;
    }

    public long flush()
    {
        _lock.lock();
        try
        {
            // Having the lock means we've sent all data in this
            //   TCP implementation.
            return 0;
        }
        finally
        {
            _lock.unlock();
        }
    }

    public long flush(long timeout)
    {
        _lock.lock();
        try
        {
            // Having the lock means we've sent all data in this
            //   TCP implementation.
            return 0;
        }
        finally
        {
            _lock.unlock();
        }
    }

    public void handleCloseEvent(int failedVersion, String message, Exception e_) throws RetryOperationException, DisconnectedException
    {
        // Let all waiting clients know that this version of the connection is dead, if they didn't already know.
        _onDisconnect.preInvoke(failedVersion);

        //if readerThread is null, we were never fully initialized
        if((_readerThread != null) && !_readerThread.equals(Thread.currentThread()))
        {
            _lock.lock();
        }
        else
        {
            // The reader thread can cause a deadlock if send thread first grabs
            // the lock and then reader thread blocks trying to acquire it before
            // getting interrupted by the send thread.
            try
            {
                while (!_lock.tryLock(100, TimeUnit.MILLISECONDS))
                {
                    if (Thread.currentThread().isInterrupted())
                        throw new DisconnectedException("Reconnect is in progress in send thread.");
                }
            }
            catch(InterruptedException e)
            {
                throw new DisconnectedException("Reconnect already in progress in send thread.");
            }
        }
        try
        {
            // Don't try to reconnect if a disconnect is in progress.
            if(_disconnecting)
            {
                throw new DisconnectedException("Disconnect in progress.");
            }
            // If there's a new version of the connection available, you should use that.
            if(failedVersion != _connectionVersion)
            {
                throw new RetryOperationException("A new connection is available.");
            }

            //OK, our thread is in charge of disconnecting and reconnecting. Let's do it.
            try
            {
                // forget about any SO_LINGER we might have going; we want to kill this connection now.
                // this could fail, based on the underlying state of the socket.  Just ignore that --
                // if it fails, that means we're not going to have to wait for anything in _disconnect either.
                _socket.socket().setSoLinger(true, 0);
            }
            catch(Exception ex)
            {
            }

            try
            {
                _disconnect();
            }
            catch (Exception ex)
            {
            }

            // Create a wrapper around ourselves to pass into the disconnect handler.
            TCPTransport t = TCPTransport.createTransport(this._messageType);
            t._impl = this;
            try
            {
                _onDisconnect.invoke(t, new DisconnectedException(message, e_));
            }
            catch (Exception e)
            {
                throw new DisconnectedException("Disconnect handler threw an exception", e);
            }
        }
        finally
        {
            _lock.unlock();
        }
        // This doesn't need to be locked, because we only care if the connection version is
        // something other than what we arrived with.
        if(_connectionVersion == failedVersion)
        {
            // no work was done.  bail.
            throw new DisconnectedException("A disconnect occured, and no disconnect handler successfuly reconnected.");
        }
        else
        {
            throw new RetryOperationException("Reconnect successful; retry the operation.");
        }
    }
    static class TCPReaderThread extends Thread
    {
        TCPTransportImpl transport   = null;
        Protocol      messageType = null;
        boolean          stopped     = false;

        TCPReaderThread(TCPTransportImpl transport, Protocol messageType)
        {
            this.transport   = transport;
            this.messageType = messageType;
            this.stopped = false;
            this.setDaemon(TCPTransport.isDaemon());
            this.start();
        }

        public void stopThread()
        {
            this.stopped = true;
        }

        @Override
        public void run()
        {
            this.setName(String.format("AMPS Java Client Background Reader Thread %d", Thread.currentThread().getId()));
            ByteBuffer rfifo = ByteBuffer.allocate(16 * 1024);
            ProtocolParser protocolParser = this.messageType.getMessageStream();
            Selector selector = this.transport._selector;
            while (true)
            {
                // This won't stop us if we're blocked on the socket read
                if(this.stopped) return;

                try
                {
                    // remember the connection version we attempted to read with.
                    int currentVersion = this.transport._connectionVersion;
                    try
                    {
                        // read more data into the rfifo from the non-blocking socket.
                        int readBytes = this.transport._socket.read(rfifo);

                        // -1 is returned from non-blocking read on end of stream.
                        if(readBytes == -1)
                        {
                            String message = "The remote server has closed the connection.";
                            transport.handleCloseEvent(currentVersion,
                                    message,
                                    new DisconnectedException(message));
                            return;
                        }

                        // no data; take the hit and go select().
                        if(readBytes == 0)
                        {
                            // clear out the key set from last time.
                            Iterator keyIterator = selector.selectedKeys().iterator();
                            while(keyIterator.hasNext())
                            {
                                keyIterator.next();
                                keyIterator.remove();
                            }
                            transport._socket.register(selector, SelectionKey.OP_READ);
                            // java select has the same semantics we do: 0 means wait forever.
                            selector.select(transport._readTimeout);
                            if(transport._readTimeout != 0 && selector.selectedKeys().size() == 0)
                            {
                                String message = String.format("No activity after %d milliseconds; connection closed by client.",
                                                               transport._readTimeout);
                                transport.handleCloseEvent(currentVersion,
                                        message,
                                        new DisconnectedException(message));
                                return;
                            }
                            else
                            {
                                // try reading again if there are selected keys, or if there are *no* selected keys,
                                // but there's also no timeout set. In that funny case, select has returned, but there's
                                // nothing available to read, and we don't really know why it returned yet. Making the
                                // choice to just go read again also allows us to bypass calling selectedKeys() when
                                // no timeout is set.
                                continue;
                            }
                        }
                    }
                    catch (java.nio.channels.ClosedSelectorException cse)
                    {
                        // this occurs when we explicitly close the selector, part of a normal disconnect.
                        return;
                    }
                    catch (IOException ioex)
                    {
                        // only handle the close event if stopThead was not invoked
                        // we don't want to invoke the disconnectHandler if we intentionally
                        // try to disconnect
                        if(!this.stopped)
                        {
                            this.transport.handleCloseEvent(currentVersion, "Exception while reading", ioex);
                        }
                        return;
                    }

                    rfifo.flip(); // Move into get-mode

                    if(rfifo.remaining() >= 4)
                    {
                        // We read data into readBuffer
                        int size = rfifo.getInt(rfifo.position());
                        this.transport._filter.incoming(rfifo);
                        // Process until we need to fetch more data
                        while (rfifo.remaining() - 4 >= size)
                        {
                            // Best-case: We have everything in the buffer
                            size = rfifo.getInt();
                            try
                            {
                                protocolParser.process(rfifo, size, this.transport._onMessage);
                            }
                            catch(Exception e)
                            {
                                if(transport._exceptionListener != null)
                                {
                                    transport._exceptionListener.exceptionThrown(e);
                                }
                            }

                            // Fetch next size, if possible
                            if(rfifo.remaining() < 4)
                                break;
                            size = rfifo.getInt(rfifo.position());
                        }

                        // We need to prepare for fetching more data
                        if(rfifo.capacity() < size + 4)
                        {
                            // Worst-case: Our buffer isn't big enough to hold the
                            // message, let's resize.
                            int newSize = rfifo.capacity();
                            while (newSize < size + 4) newSize *= 2; // double until big enough
                            // System.out.println("size: " + size);
                            // System.out.println("newSize: " + newSize);
                            ByteBuffer newBuffer = ByteBuffer.allocate(newSize);
                            newBuffer.put(rfifo);
                            rfifo = newBuffer;
                        }
                        else rfifo.compact();
                    }
                    else rfifo.compact();

                }
                catch (Exception e)
                {
                    if(!this.stopped && transport._exceptionListener != null)
                    {
                        transport._exceptionListener.exceptionThrown(e);
                    }
                }
            }
        }
    }

    public void setReadTimeout(int readTimeoutMillis_)
    {
        _readTimeout = readTimeoutMillis_;

    }
}
