////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.client;

import java.lang.Class;
import java.lang.reflect.Constructor;
import java.util.Hashtable;
import java.util.Properties;

import com.crankuptheamps.client.exception.TransportTypeException;

public class TransportFactory
{

    private static Hashtable<String, String> registry= new Hashtable<String, String>();

    /**
     * Registers a new {@link Transport} and URI prefix with the AMPS Client.
     * 
     * @param name The URI prefix to register for this transport.
     * @param className The name of the Java class to instantiate.
     * @throws TransportTypeException Thrown when this transport type is already registered.
     */
    public static void register(
        String name,
        String className) throws TransportTypeException
    {
        if(!registry.containsKey(name))
        {
            registry.put(name, className);
        }
        else
            throw new TransportTypeException("'" + name + "' is already registered");
    }
    /**
     * Unregisters a {@link Transport} from the AMPS client.
     * <b>Note</b>: built-in transport types (e.g. `tcp') cannot be unregistered.
     * 
     * @param name The URI prefix to unregister.
     * @throws TransportTypeException Thrown when this transport type has not been registered.
     */
    public static void unregister(String name) throws TransportTypeException
    {
        if (registry.containsKey(name))
        {
            registry.remove(name);
        }
        else
        {
            throw new TransportTypeException("The TransportType '"  + name + "' is not a known TransportType");
        }
    }

    /**
     * Resets self to the default state, unregistering any {@link Transport}s registered at runtime.
     * 
     */
    public static void reset()
    {
        registry.clear();
    }
    public static Transport createTransport(
        String name,
        Protocol messageType,
        Properties props) throws TransportTypeException
    {
        // Whoops, we don't have a Transport by this name in the registry,
        //   let's guess at the correct type by the name.
        String tName = "com.crankuptheamps.client." + name.toUpperCase() + "Transport";
        if (TransportFactory.registry.containsKey(name))
        {
            tName = TransportFactory.registry.get(name);
        }
        try
        {
            Class<?> c = Class.forName(tName);
            Class[] params = new Class[2];
            params[0] = Protocol.class;
            params[1] = Properties.class;
            Constructor ctor = c.getConstructor(params);
            Object[] args = new Object[2];
            args[0] = messageType;
            args[1] = props;
            Transport result = (Transport)ctor.newInstance(args);
            return result;
        }
        catch(Exception e)
        {
            throw new TransportTypeException("The TransportType '" + name + "' could not be found.");
        }
    }

}
