////////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2010-2015 60East Technologies Inc., All Rights Reserved.
//
// This computer software is owned by 60East Technologies Inc. and is
// protected by U.S. copyright laws and other laws and by international
// treaties.  This computer software is furnished by 60East Technologies
// Inc. pursuant to a written license agreement and may be used, copied,
// transmitted, and stored only in accordance with the terms of such
// license agreement and with the inclusion of the above copyright notice.
// This computer software or any other copies thereof may not be provided
// or otherwise made available to any other person.
//
// U.S. Government Restricted Rights.  This computer software: (a) was
// developed at private expense and is in all respects the proprietary
// information of 60East Technologies Inc.; (b) was not developed with
// government funds; (c) is a trade secret of 60East Technologies Inc.
// for all purposes of the Freedom of Information Act; and (d) is a
// commercial item and thus, pursuant to Section 12.212 of the Federal
// Acquisition Regulations (FAR) and DFAR Supplement Section 227.7202,
// Government's use, duplication or disclosure of the computer software
// is subject to the restrictions set forth by 60East Technologies Inc..
//
////////////////////////////////////////////////////////////////////////////

package com.crankuptheamps.spark;

import java.util.UUID;

import com.crankuptheamps.client.Client;
import com.crankuptheamps.client.ClientDisconnectHandler;
import com.crankuptheamps.client.exception.AMPSException;
import com.crankuptheamps.client.exception.ConnectionException;

public class PingCommand extends SparkCommand
{
    private String uriString    = null;

    public PingCommand(String name)
    {
        super(name);
        addOption(new CommandOption("verbose", "prints time to connect and logon to stderr", false));
        
        // These are the examples...
        StringBuilder example = new StringBuilder(1024);
        example.append("  % ./spark ping -server localhost:9003 -proto amps\n\n");
        example.append("    Connects to the AMPS instance listening on port 9003 using the amps protocol and attempts to logon.\n");

        addExample(example.toString());
    }

    
    private class SparkDisconnectHandler implements ClientDisconnectHandler
    {
        public void invoke(Client client)
        {
            System.err.format("DISCONNECTED, trying to reconnect...%n");
            try
            {
                client.connect(uriString);
                client.logon(5000, createAuthenticator(uriString));
            }
            catch (AMPSException e)
            {
                System.err.println("Unable to re-connect to AMPS (" + e + ").");
                System.exit(1);
            }
        }
    }

    void execute(String argv[])
    {
        // Extract all of the command line arguments
        parseArgs(argv, 1);
        String[] userAndServer = options.get("server").getValue().split("@",2);
        String server = (userAndServer.length == 2) ? userAndServer[1] : userAndServer[0];
        String type          = options.get("proto").getValue();
        
        if(!type.equals("xml") && !type.equals("fix") && !type.equals("nvfix") && !type.equals("amps") && !type.equals("json"))
        {
            badUsage("prot argument needs to be 'fix', 'xml', 'nvfix', 'amps', or 'json'.");
        }

        // Connect client to AMPS
        String userId = (userAndServer.length == 2) ? userAndServer[0] : System.getProperty("user.name");
        String clientName = "spark-subscribe-" + userId + "-" + UUID.randomUUID().toString();
        Client client = new Client(clientName, Client.Version.AMPS_2);
        client.setDisconnectHandler(new SparkDisconnectHandler());
        long start = System.currentTimeMillis();
        try
        {
            uriString = "tcp://" + userId + "@" + server + "/" + type;
            client.connect(uriString);
            long connectTime = System.currentTimeMillis() - start;
            client.logon(5000, createAuthenticator(uriString));
            long logonTime = System.currentTimeMillis() - start - connectTime;
            System.out.println("Successfully connected to " + uriString);
            if(options.get("verbose").hasValue())
                System.err.format("Total time: %d ms (connection took %d ms, logon took %d ms.)\n", connectTime+logonTime, connectTime, logonTime);
        }
        catch (ConnectionException e)
        {
            System.err.println("Unable to connect to AMPS (" + e.toString().trim() + ").");
            System.exit(1);
        }
        client.close();
    }
}
